

#ifndef SAMPLING_H
#define SAMPLING_H

#include "ShaderCore.h"



struct Sampler
{
    uint    m_index;        
    uint    m_scramble;     
    uint    m_dimension;    
};










void GetOrthoVectors(in vec3 n, out vec3 b1, out vec3 b2)
{
    const float sign = (n.z >= 0.0f ? 1.0f : -1.0f);
    const float a = -1.0f / (sign + n.z);
    const float b = n.x * n.y * a;
    b1 = vec3(1.0f + sign * n.x * n.x * a, sign * b, -sign * n.x);
    b2 = vec3(b, sign + n.y * n.y * a, -n.y);
}







mat3 CreateTBN(in vec3 N)
{
    vec3 U, V;
    GetOrthoVectors(N, U, V);
    return transpose(mat3(U, V, N));
}







mat3 CreateTBNInverse(in vec3 N)
{
    vec3 U, V;
    GetOrthoVectors(N, U, V);
    return mat3(U, V, N);
}









vec3 MapToHemisphere(in vec2 s, in vec3 n, in float e)
{
    
    vec3 u, v;
    GetOrthoVectors(n, u, v);

    
    const float r1 = s.x;
    const float r2 = s.y;

    
    const float sinpsi = sin(2.0f * FW_PI * r1);
    const float cospsi = cos(2.0f * FW_PI * r1);
    const float costheta = pow(1.0f - r2, 1.0f / (e + 1.0f));
    const float sintheta = sqrt(1.0f - costheta * costheta);

    
    return normalize(u * sintheta * cospsi + v * sintheta * sinpsi + n * costheta);
}








vec3 MapToHemisphere(in vec2 s, in vec3 n)
{
    return MapToHemisphere(s, n, 0.0f);
}







vec2 MapToDisk(in vec2 s)
{
    const float r = sqrt(s.x);
    const float theta = 2.0f * FW_PI * s.y;
    return vec2(r * cos(theta), r * sin(theta));
}







vec3 MapToSphere(in vec2 s)
{
    const float theta = 2.0f * FW_PI * s.x;
    const float phi = acos(1.0f - 2.0f * s.y);
    const float x = sin(phi) * cos(theta);
    const float y = sin(phi) * sin(theta);
    const float z = cos(phi);
    return vec3(x, y, z);
}







vec3 MapToSphere(in vec3 s)
{
    const float theta = 2.0f * FW_PI * s.x;
    const float phi = acos(1.0f - 2.0f * s.y);
    const float x = sin(phi) * cos(theta);
    const float y = sin(phi) * sin(theta);
    const float z = cos(phi);
    return vec3(x, y, z) * pow(s.z, 1.0f / 3.0f);
}







vec2 MapToSphereInverse(in vec3 p)
{
    const float tmp = atan(p.y, p.x);
    const float theta = (tmp < 0.0f ? (tmp + 2.0f * FW_PI) : tmp);
    const float x = theta / (2.0f * FW_PI);
    const float y = (1.0f - p.z) / 2.0f;
    return vec2(x, y);
}







vec2 MapToTriangle(in vec2 s)
{
    vec2 uv;
    uv.x = sqrt(s.x);
    uv.y = s.y * uv.x;
    uv.x = 1.0f - uv.x;
    return uv;
}









float D(in vec3 Ne, in float alpha_x, in float alpha_y)
{
    const float x2 = Square(Ne.x / alpha_x),
                y2 = Square(Ne.y / alpha_y),
                z2 = Square(Ne.z);

    return 1.0f / (FW_PI * alpha_x * alpha_y * Square(x2 + y2 + z2));
}









float G1(in vec3 Xe, in float alpha_x, in float alpha_y)
{
    const float x2 = Square(alpha_x * Xe.x),
                y2 = Square(alpha_y * Xe.y),
                z2 = Square(Xe.z),
                lambda = 0.5f * (-1.0f + sqrt(1.0f + (x2 + y2) / z2));

    return 1.0f / (1.0f + lambda);
}










float G2(in vec3 Ve, in vec3 Le, in float alpha_x, in float alpha_y)
{
    return G1(Ve, alpha_x, alpha_y) * G1(Le, alpha_x, alpha_y);
}














vec4 SampleGGXVNDF(in vec3 Ve, in float alpha_x, in float alpha_y, in float U1, in float U2)
{
    
    vec3 Vh = normalize(vec3(alpha_x * Ve.x, alpha_y * Ve.y, Ve.z));
    
    float lensq = Vh.x * Vh.x + Vh.y * Vh.y;
    vec3 T1 = (lensq > 0.0f ? vec3(-Vh.y, Vh.x, 0.0f) / sqrt(lensq) : vec3(1.0f, 0.0f, 0.0f));
    vec3 T2 = cross(Vh, T1);
    
    float r = sqrt(U1);
    float phi = 2.0f * FW_PI * U2;
    float t1 = r * cos(phi);
    float t2 = r * sin(phi);
    float s = 0.5f * (1.0f + Vh.z);
    t2 = (1.0f - s) * sqrt(1.0f - t1 * t1) + s * t2;
    
    vec3 Nh = t1 * T1 + t2 * T2 + sqrt(max(0.0f, 1.0f - t1 * t1 - t2 * t2)) * Vh;
    
    vec3 Ne = normalize(vec3(alpha_x * Nh.x, alpha_y * Nh.y, max(0.0f, Nh.z)));
    
    return vec4(Ne, G1(Ve, alpha_x, alpha_y) * max(dot(Ve, Ne), 0.0f) * D(Ne, alpha_x, alpha_y) / Ve.z);
}











vec4 SampleGGXVNDFEllipsoid(in vec3 Ve, in float alpha_x, in float alpha_y, in float U1, in float U2)
{
    return SampleGGXVNDF(Ve, alpha_x, alpha_y, U1, U2);
}










vec4 SampleGGXVNDFHemisphere(in vec3 Ve, in float alpha, in float U1, in float U2)
{
    return SampleGGXVNDFEllipsoid(Ve, alpha, alpha, U1, U2);
}










vec2 SampleOctahedral(in vec3 v)
{
    
    const vec2 p = v.xy * (1.0f / (abs(v.x) + abs(v.y) + v.z));
    
    return vec2(p.x + p.y, p.x - p.y);
}










vec3 SampleOctahedralInverse(in vec2 e)
{
    const vec2 tmp = vec2(e.x + e.y, e.x - e.y) * 0.5f;
    const vec3 v = vec3(tmp, 1.0f - abs(tmp.x) - abs(tmp.y));
    return normalize(v);
}









uint Permute(in uint i, in uint l, in uint p)
{
    uint w = l - 1;
    w |= w >> 1;
    w |= w >> 2;
    w |= w >> 4;
    w |= w >> 8;
    w |= w >> 16;
    do
    {
        i ^= p;
        i *= 0xe170893d;
        i ^= p >> 16;
        i ^= (i & w) >> 4;
        i ^= p >> 8;
        i *= 0x0929eb3f;
        i ^= p >> 23;
        i ^= (i & w) >> 1;
        i *= 1 | p >> 27;
        i *= 0x6935fa69;
        i ^= (i & w) >> 11;
        i *= 0x74dcb303;
        i ^= (i & w) >> 2;
        i *= 0x9e501cc3;
        i ^= (i & w) >> 2;
        i *= 0xc860a3df;
        i &= w;
        i ^= i >> 5;
    }
    while(i >= l);
    return (i + p) % l;
}








float RandFloat(in uint i, in uint p)
{
    i ^= p;
    i ^= i >> 17;
    i ^= i >> 10;
    i *= 0xb36534e5;
    i ^= i >> 12;
    i ^= i >> 21;
    i *= 0x93fc4795;
    i ^= 0xdf6e307f;
    i ^= i >> 17;
    i *= 1 | p >> 18;
    return i * (1.0f / 4294967808.0f);
}












vec2 CMJ(in uint s, in uint n, in uint p)
{
    const uint sx = Permute(s % n, n, p * 0xa511e9b3);
    const uint sy = Permute(s / n, n, p * 0x63d83595);
    const float jx = RandFloat(s, p * 0xa399d265);
    const float jy = RandFloat(s, p * 0x711ad6a5);

    return vec2(
        (s % n + (sy + jx) / n) / n,
        (s / n + (sx + jy) / n) / n);
}







float VanDerCorput(in uint bits)
{
    bits = (bits << 16u) | (bits >> 16u);
    bits = ((bits & 0x55555555u) << 1u) | ((bits & 0xAAAAAAAAu) >> 1u);
    bits = ((bits & 0x33333333u) << 2u) | ((bits & 0xCCCCCCCCu) >> 2u);
    bits = ((bits & 0x0F0F0F0Fu) << 4u) | ((bits & 0xF0F0F0F0u) >> 4u);
    bits = ((bits & 0x00FF00FFu) << 8u) | ((bits & 0xFF00FF00u) >> 8u);
    return float(double(bits) * 2.3283064365386963e-10);    
}











vec2 Hammersley(in uint i, in uint N)
{
    return vec2(float(i) / float(N), VanDerCorput(i));
}









vec3 ImportanceSampleGGX(in vec2 Xi, in vec3 N, in float roughness)
{
    float a = roughness * roughness;

    float phi = 2.0f * FW_PI * Xi.x;
    float cosTheta = sqrt((1.0f - Xi.y) / (1.0f + (a * a - 1.0f) * Xi.y));
    float sinTheta = sqrt(1.0f - cosTheta * cosTheta);

    
    vec3 H;
    H.x = cos(phi) * sinTheta;
    H.y = sin(phi) * sinTheta;
    H.z = cosTheta;

    
    vec3 up = abs(N.z) < 0.999f ? vec3(0.0f, 0.0f, 1.0f) : vec3(1.0f, 0.0f, 0.0f);
    vec3 tangent = normalize(cross(up, N));
    vec3 bitangent = cross(N, tangent);

    return normalize(tangent * H.x + bitangent * H.y + N * H.z);
}








float CalculateHaltonNumber(in uint index, in uint base)
{
    float f = 1.0f, result = 0.0f;
    for(uint i = index; i > 0;)
    {
        f /= base;
        result = result + f * (i % base);
        i = uint(i / float(base));
    }
    return result;
}







vec2 CalculateHaltonNumber(in uint index)
{
    return vec2(CalculateHaltonNumber((index & 0xFFu) + 1, 2),
                CalculateHaltonNumber((index & 0xFFu) + 1, 3));
}










float InterleavedGradientNoise(in vec2 screenPosition)
{
    const vec3 magic = vec3(0.06711056f, 0.00583715f, 52.9829189f);
    return fract(magic.z * fract(dot(screenPosition, magic.xy)));
}












vec2 VogelDiskSample(in uint sampleIndex, in uint sampleCount, in float angle)
{
    const float theta = sampleIndex * FW_GOLDEN_ANGLE + angle;
    const float r = sqrt(sampleIndex + 0.5f) / sqrt(sampleCount);
    return vec2(r * cos(theta), r * sin(theta));
}

















float SampleBlueNoise(in uint pixel_i, in uint pixel_j, in uint sampleIndex, in uint sampleDimension)
{
    
    pixel_i = pixel_i & 127;
    pixel_j = pixel_j & 127;
    sampleIndex = sampleIndex & 255;
    sampleDimension = sampleDimension & 255;

    
    uint rankedSampleIndex = sampleIndex ^ texelFetch(RankingTileBuffer, int(sampleDimension + (pixel_i + pixel_j * 128) * 8)).x;

    
    uint value = texelFetch(SobolBuffer, int(sampleDimension + rankedSampleIndex * 256)).x;

    
    value = value ^ texelFetch(ScramblingTileBuffer, int((sampleDimension % 8) + (pixel_i + pixel_j * 128) * 8)).x;

    
    return (0.5f + value) / 256.0f;
}









float Sample1D(in uvec2 pixelCoords, in uint sampleIndex, in uint dimensionOffset)
{
    const float s = SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset);

    return mod(s + (sampleIndex & 255) * FW_GOLDEN_RATIO, 1.0f);
}








float Sample1D(in uvec2 pixelCoords, in uint sampleIndex)
{
    return Sample1D(pixelCoords, sampleIndex, 0);
}









vec2 Sample2D(in uvec2 pixelCoords, in uint sampleIndex, in uint dimensionOffset)
{
    const vec2 s = vec2(SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset + 0),
                        SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset + 1));

    return mod(s + (sampleIndex & 255) * FW_GOLDEN_RATIO, 1.0f);
}








vec2 Sample2D(in uvec2 pixelCoords, in uint sampleIndex)
{
    return Sample2D(pixelCoords, sampleIndex, 0);
}









vec3 Sample3D(in uvec2 pixelCoords, in uint sampleIndex, in uint dimensionOffset)
{
    const vec3 s = vec3(SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset + 0),
                        SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset + 1),
                        SampleBlueNoise(pixelCoords.x, pixelCoords.y, 0, dimensionOffset + 2));

    return mod(s + (sampleIndex & 255) * FW_GOLDEN_RATIO, 1.0f);
}








vec3 Sample3D(in uvec2 pixelCoords, in uint sampleIndex)
{
    return Sample3D(pixelCoords, sampleIndex, 0);
}














vec4 SampleReflectionVector(in vec3 viewDirection, in vec3 normal, in float roughness, in uvec2 pos)
{
    const mat3 tbn = CreateTBN(normal);
    const vec3 viewDirectionTBN = -tbn * viewDirection;
    const vec2 u = Sample2D(pos, Frame);    
    const vec4 sampledNormalTBN = SampleGGXVNDFHemisphere(viewDirectionTBN, roughness, u.x, u.y);
    const vec3 reflectedDirectionTBN = reflect(-viewDirectionTBN, sampledNormalTBN.xyz);
    const mat3 tbnInverse = transpose(tbn); 
    return vec4(tbnInverse * reflectedDirectionTBN, sampledNormalTBN.w);
}













Sampler SamplerInit(in uint index, in uint frame, in uint offset)
{
    Sampler sampler;
    #define CMJ_DIM 16u
    const uint rnd = texelFetch(RandomBuffer, int(index)).x;
    const uint scramble = rnd * 0x1fe3434f * ((frame + 331 * rnd) / (CMJ_DIM * CMJ_DIM));
    sampler.m_index = frame % (CMJ_DIM * CMJ_DIM);
    sampler.m_dimension = 300 * offset + 5;
    sampler.m_scramble = scramble;
    return sampler;
}








Sampler SamplerInit(in uvec2 pixel, in uint offset)
{
    return SamplerInit(pixel.x + uint(ViewportWidthHeight.x) * pixel.y, Frame, offset);
}








Sampler SamplerInit(in uint index, in uint offset)
{
    return SamplerInit(index, Frame, offset);
}







Sampler SamplerInit(in uvec2 pixel)
{
    return SamplerInit(pixel.x + uint(ViewportWidthHeight.x) * pixel.y, Frame, 0);
}







Sampler SamplerInit(in uint index)
{
    return SamplerInit(index, Frame, 0);
}







float Sample1D(inout Sampler sampler)
{
    vec2 s;
    const uint idx = Permute(sampler.m_index, CMJ_DIM * CMJ_DIM, 0xa399d265 * sampler.m_dimension * sampler.m_scramble);
    s = CMJ(idx, CMJ_DIM, sampler.m_dimension * sampler.m_scramble);
    ++sampler.m_dimension;
    return s.x;
}







vec2 Sample2D(inout Sampler sampler)
{
    vec2 s;
    const uint idx = Permute(sampler.m_index, CMJ_DIM * CMJ_DIM, 0xa399d265 * sampler.m_dimension * sampler.m_scramble);
    s = CMJ(idx, CMJ_DIM, sampler.m_dimension * sampler.m_scramble);
    ++sampler.m_dimension;
    return s;
}







vec3 Sample3D(inout Sampler sampler)
{
    return vec3(
        Sample2D(sampler),
        Sample1D(sampler));
}

#endif 
