

#ifndef SHADER_CORE_H
#define SHADER_CORE_H

#include "ShaderSharedCode.h"







#define FW_E                2.71828182845904523536f



#define FW_PI               3.14159265358979323846f



#define FW_TAU              (2.0f * FW_PI)



#define FW_FLT_MAX          3.402823466e+38f






#define FW_GOLDEN_RATIO     1.61803398874989484820f






#define FW_GOLDEN_ANGLE     2.39996322972865332f



#define FW_LTC_LUT_SIZE     64.0f



#define FW_LTC_LUT_SCALE    ((FW_LTC_LUT_SIZE - 1.0f) / FW_LTC_LUT_SIZE)



#define FW_LTC_LUT_BIAS     (0.5f / FW_LTC_LUT_SIZE)







#define iDispatchId     int(DispatchId)



#define uDispatchId     DispatchId



#define iDrawId         DrawId



#define uDrawId         uint(DrawId)



#define iLocalId        ivec3(gl_LocalInvocationID)



#define uLocalId        gl_LocalInvocationID



#define iLocalIndex     int(gl_LocalInvocationID.x + gl_LocalInvocationID.y * gl_WorkGroupSize.x + gl_LocalInvocationID.z * gl_WorkGroupSize.x * gl_WorkGroupSize.y)



#define uLocalIndex     (gl_LocalInvocationID.x + gl_LocalInvocationID.y * gl_WorkGroupSize.x + gl_LocalInvocationID.z * gl_WorkGroupSize.x * gl_WorkGroupSize.y)



#define iGlobalId       ivec3(gl_GlobalInvocationID)



#define uGlobalId       gl_GlobalInvocationID



#define iGroupId        ivec3(gl_WorkGroupID)



#define uGroupId        gl_WorkGroupID



#define iGroupCount     ivec3(gl_NumWorkGroups)



#define uGroupCount     gl_NumWorkGroups



#define iGroupSize      ivec3(gl_WorkGroupSize)



#define uGroupSize      gl_WorkGroupSize



#define uPixel          uvec2(gl_FragCoord.xy - ViewportXY)



#define iPixel          ivec2(gl_FragCoord.xy - ViewportXY)



#define iPixelId        (int(gl_FragCoord.x) + int(gl_FragCoord.y) * int(ViewportWidthHeight.x))



#define uPixelId        (uint(gl_FragCoord.x) + uint(gl_FragCoord.y) * uint(ViewportWidthHeight.x))





uniform vec3                    Eye;                                    
uniform uint                    Frame;                                  
uniform uint                    DispatchId;                             
uniform mat4                    Reprojection;                           

uniform mat4                    View;                                   
uniform mat4                    ViewInverse;                            
uniform mat4                    PreviousView;                           
uniform mat4                    PreviousViewInverse;                    
uniform mat4                    Projection;                             
uniform mat4                    ProjectionInverse;                      
uniform mat4                    PreviousProjection;                     
uniform mat4                    PreviousProjectionInverse;              
uniform mat4                    ViewProjection;                         
uniform mat4                    ViewProjectionInverse;                  
uniform mat4                    PreviousViewProjection;                 
uniform mat4                    PreviousViewProjectionInverse;          

uniform float                   Aspect;                                 
uniform float                   FovY;                                   
uniform vec2                    NearFar;                                
uniform float                   NearTimesFar;                           
uniform float                   FarMinusNear;                           
uniform float                   FarMinusNearInverse;                    
uniform vec2                    ScreenWidthHeight;                      
uniform vec2                    ScreenWidthHeightInverse;               
uniform vec2                    ViewportXY;                             
uniform vec2                    ViewportWidthHeight;                    
uniform vec2                    ViewportWidthHeightInverse;             

uniform samplerBuffer           LightBuffer;                            
uniform samplerBuffer           AreaLightBuffer;                        
uniform usamplerBuffer          LightListBuffer;                        
uniform sampler2D               BrdfBuffer;                             
uniform sampler2D               LtcAmpBuffer;                           
uniform sampler2D               LtcMatBuffer;                           
uniform samplerCube             EnvironmentBuffer;                      
uniform samplerCube             IrradianceBuffer;                       
uniform float                   SkylightIntensity;                      
uniform samplerBuffer           MaterialBuffer;                         
uniform usamplerBuffer          TileBuffer;                             
uniform uint                    MaxLightCountPerTile;                   
uniform uvec2                   TileCount;                              
uniform usamplerBuffer          RandomBuffer;                           
uniform sampler2D               DiffuseBuffer;                          
uniform sampler2D               OcclusionBuffer;                        
uniform usamplerBuffer          RankingTileBuffer;                      
uniform usamplerBuffer          ScramblingTileBuffer;                   
uniform usamplerBuffer          SobolBuffer;                            
uniform sampler2DArray          ShadowBuffer;                           
uniform samplerBuffer           ShadowSplitBuffer;                      
uniform samplerBuffer           ShadowMatrixBuffer;                     
uniform sampler2DArrayShadow    ShadowAtlas[FW_SHADOW_ATLAS_SIZE];      
uniform samplerBuffer           BufferCache[FW_BUFFER_CACHE_SIZE];      
uniform sampler2DArray          TextureCache[FW_TEXTURE_CACHE_SIZE];    
uniform sampler3D               VolumeCache[FW_VOLUME_CACHE_SIZE];      
uniform usamplerBuffer          ZBinBuffer;                             







struct LightIterator
{
    uint    m_lightIndex;       
    uint    m_lightCursor;      
    uint    m_lightEnd;         
    uint    m_zBin;             
};



struct Material
{
    vec3    m_albedo;           
    vec3    m_emissivity;       
    vec3    m_fresnelBase;      
    vec3    m_normal;           
    float   m_metallicity;      
    float   m_roughness;        
    float   m_ao;               
};



struct Surface
{
    vec3    m_normal;           
    vec2    m_texcoords;        
    vec3    m_objectPosition;   
    vec3    m_worldPosition;    
    vec3    m_shadingLayer;     
};











float Square(in float value)
{
    return value * value;
}






bool HasJitter()
{
    return (Projection[2][0] != 0.0f || Projection[2][1] != 0.0f ? true : false);
}





vec3 GetForward()
{
    return -vec3(View[0][2], View[1][2], View[2][2]);
}





vec3 GetSide()
{
    return vec3(View[0][0], View[1][0], View[2][0]);
}





vec3 GetUp()
{
    return vec3(View[0][1], View[1][1], View[2][1]);
}







mat2 Rotate2D(in float angle)
{
    return mat2(cos(angle), -sin(angle),
                sin(angle),  cos(angle));
}







float RemoveNaNs(in float color)
{
    color /= (1.0f + color);
    color  = clamp(color, 0.0f, 1.0f);
    color /= max(1.0f - color, 1e-3f);
    return color;
}







vec3 RemoveNaNs(in vec3 color)
{
    color /= (1.0f + color);
    color  = clamp(color, 0.0f, 1.0f);
    color /= max(1.0f - color, 1e-3f);
    return color;
}







vec4 RemoveNaNs(in vec4 color)
{
    color /= (1.0f + color);
    color  = clamp(color, 0.0f, 1.0f);
    color /= max(1.0f - color, 1e-3f);
    return color;
}







vec3 GetPassthruColor(in vec3 color)
{
    return color * color;
}







float GetLinearDepth(in float depth)
{
#ifdef USE_ORTHO
    return -(depth * FarMinusNear + NearFar.x);
#else 
    return -NearTimesFar / (depth * FarMinusNear - NearFar.y);
#endif 
}







mat4 GetTransform(in MeshData meshData)
{
    return mat4(
        vec4(meshData.m_matrixCol0.xyz, 0.0f),
        vec4(meshData.m_matrixCol1.xyz, 0.0f),
        vec4(meshData.m_matrixCol2.xyz, 0.0f),
        vec4(meshData.m_matrixCol0.w,
             meshData.m_matrixCol1.w,
             meshData.m_matrixCol2.w, 1.0f));
}











vec3 GetTransformedNormal(in mat4 transform, in vec3 normal)
{
    mat4 matrix;
    #define minor(m, r0, r1, r2, c0, c1, c2)                            \
        (m[c0][r0] * (m[c1][r1] * m[c2][r2] - m[c1][r2] * m[c2][r1]) -  \
         m[c1][r0] * (m[c0][r1] * m[c2][r2] - m[c0][r2] * m[c2][r1]) +  \
         m[c2][r0] * (m[c0][r1] * m[c1][r2] - m[c0][r2] * m[c1][r1]))
    matrix[0][0] =  minor(transform, 1, 2, 3, 1, 2, 3);
    matrix[1][0] = -minor(transform, 1, 2, 3, 0, 2, 3);
    matrix[2][0] =  minor(transform, 1, 2, 3, 0, 1, 3);
    matrix[3][0] = -minor(transform, 1, 2, 3, 0, 1, 2);
    matrix[0][1] = -minor(transform, 0, 2, 3, 1, 2, 3);
    matrix[1][1] =  minor(transform, 0, 2, 3, 0, 2, 3);
    matrix[2][1] = -minor(transform, 0, 2, 3, 0, 1, 3);
    matrix[3][1] =  minor(transform, 0, 2, 3, 0, 1, 2);
    matrix[0][2] =  minor(transform, 0, 1, 3, 1, 2, 3);
    matrix[1][2] = -minor(transform, 0, 1, 3, 0, 2, 3);
    matrix[2][2] =  minor(transform, 0, 1, 3, 0, 1, 3);
    matrix[3][2] = -minor(transform, 0, 1, 3, 0, 1, 2);
    matrix[0][3] = -minor(transform, 0, 1, 2, 1, 2, 3);
    matrix[1][3] =  minor(transform, 0, 1, 2, 0, 2, 3);
    matrix[2][3] = -minor(transform, 0, 1, 2, 0, 1, 3);
    matrix[3][3] =  minor(transform, 0, 1, 2, 0, 1, 2);
    return (matrix * vec4(normal, 0.0f)).xyz;
    #undef minor    
}








mat4 GetShadowMatrix(in uint shadowBufferIndex, in uint shadowMapIndex)
{
    const int shadowMatrixIndex = int((6 * shadowBufferIndex + shadowMapIndex) << 2);
    return mat4(texelFetch(ShadowMatrixBuffer, shadowMatrixIndex + 0),
                texelFetch(ShadowMatrixBuffer, shadowMatrixIndex + 1),
                texelFetch(ShadowMatrixBuffer, shadowMatrixIndex + 2),
                texelFetch(ShadowMatrixBuffer, shadowMatrixIndex + 3));
}







uint GetMaterialId(in MeshData meshData)
{
    return floatBitsToUint(meshData.m_meshData.x);
}







vec2 GetScreenPosition(in vec2 uv)
{
    return 2.0f * uv - 1.0f;
}








vec3 GetViewPosition(in vec2 screenPosition, in float depth)
{
    vec4 viewPosition = ProjectionInverse * vec4(screenPosition, 2.0f * depth - 1.0f, 1.0f);
    return viewPosition.xyz / viewPosition.w;   
}







vec3 GetWorldPosition(in vec3 viewPosition)
{
    return (ViewInverse * vec4(viewPosition, 1.0f)).xyz;
}








vec3 GetWorldPosition(in vec2 screenPosition, in float depth)
{
    vec4 worldPosition = ViewProjectionInverse * vec4(screenPosition, 2.0f * depth - 1.0f, 1.0f);
    return worldPosition.xyz / worldPosition.w; 
}








uvec2 GetSubsampleOffset(in uvec2 coords, in uvec2 scale)
{
    uint seed = Frame * 3333 + (coords.x * 1337 + coords.y);
    seed %= (scale.x * scale.y);    
    return uvec2(seed % scale.x, seed / scale.x);
}








ivec2 GetSubsampleOffset(in ivec2 coords, in ivec2 scale)
{
    uint seed = Frame * 3333 + (coords.x * 1337 + coords.y);
    seed %= uint(scale.x * scale.y);    
    return ivec2(seed % scale.x, seed / scale.x);
}







uvec2 GetSubsampleScale(in uvec2 imageSize)
{
    return max(uvec2(ScreenWidthHeight / vec2(imageSize) + 0.5f), 1);
}







ivec2 GetSubsampleScale(in ivec2 imageSize)
{
    return max(ivec2(ScreenWidthHeight / vec2(imageSize) + 0.5f), 1);
}









vec4 Load(in uint textureId, in uvec2 coords, in uint lod)
{
    if((textureId >> 16) == 0xFFFFu)
        return vec4(0.0f, 0.0f, 0.0f, 1.0f);    
    return texelFetch(TextureCache[textureId >> 16], ivec3(coords, textureId & 0xFFFFu), int(lod));
}








vec4 Sample(in uint textureId, in vec2 uv)
{
    
    
    
    
    
    
    if((textureId >> 16) == 0xFFFFu)
        return vec4(0.0f, 0.0f, 0.0f, 1.0f);    
#ifndef FW_AMD
    return texture(TextureCache[textureId >> 16], vec3(uv, float(textureId & 0xFFFFu)));
#else 
#ifndef FW_FRAGMENT_STAGE
    return texture(TextureCache[textureId >> 16], vec3(uv, float(textureId & 0xFFFFu)));
#else 
    return textureGrad(TextureCache[textureId >> 16], vec3(uv, float(textureId & 0xFFFFu)), dFdx(uv), dFdy(uv));
#endif 
#endif 
}










vec4 SampleGrad(in uint textureId, in vec2 uv, in vec2 dPdx, in vec2 dPdy)
{
    if((textureId >> 16) == 0xFFFFu)
        return vec4(0.0f, 0.0f, 0.0f, 1.0f);    
    return textureGrad(TextureCache[textureId >> 16], vec3(uv, float(textureId & 0xFFFFu)), dPdx, dPdy);
}









vec4 SampleLevel(in uint textureId, in vec2 uv, in float lod)
{
    if((textureId >> 16) == 0xFFFFu)
        return vec4(0.0f, 0.0f, 0.0f, 1.0f);    
    return textureLod(TextureCache[textureId >> 16], vec3(uv, float(textureId & 0xFFFFu)), lod);
}








uvec2 TextureSize(in uint textureId, in uint lod)
{
    if((textureId >> 16) == 0xFFFFu)
        return uvec2(0);    
    return uvec2(textureSize(TextureCache[textureId >> 16], int(lod)).xy);
}







LightIterator LightBegin(in vec3 position)
{
    LightIterator lightIt;

    
    const ivec2 tileCoordinates = ivec2(position.xy * ViewportWidthHeight / float(FW_TILE_SIZE));
    const int tileIndex = tileCoordinates.x + tileCoordinates.y * int(TileCount.x);

    
    lightIt.m_lightCursor = tileIndex * MaxLightCountPerTile;
    lightIt.m_lightEnd = lightIt.m_lightCursor + texelFetch(TileBuffer, tileIndex).x;

    
#ifndef USE_Z_BINS
    lightIt.m_lightIndex = (texelFetch(LightListBuffer, int(lightIt.m_lightCursor)).x & 0x7fffffffu);
    lightIt.m_zBin = 0xffffu;   
#else 
    
    const float linearDepth = GetLinearDepth(position.z),
                quadraticDepth = sqrt((linearDepth - NearFar.x) * FarMinusNearInverse);
    const int zBinIndex = int(quadraticDepth * float(FW_BIN_COUNT));
    lightIt.m_zBin = texelFetch(ZBinBuffer, zBinIndex).x;

    
    for(;;)
    {
        if(lightIt.m_lightCursor >= lightIt.m_lightEnd)
            break;  
        lightIt.m_lightIndex = texelFetch(LightListBuffer, int(lightIt.m_lightCursor)).x;
        if((lightIt.m_lightIndex & 0x80000000u) != 0)
        {
            lightIt.m_lightIndex &= 0x7fffffffu;
            break;  
        }
        if(lightIt.m_lightIndex >= (lightIt.m_zBin >> 16) && lightIt.m_lightIndex <= (lightIt.m_zBin & 0xffffu))
            break;  
        ++lightIt.m_lightCursor;
    }
#endif 

    return lightIt;
}

#ifdef FW_FRAGMENT_STAGE





LightIterator LightBegin()
{
    return LightBegin(vec3(gl_FragCoord.xy * ViewportWidthHeightInverse, gl_FragCoord.z));
}

#endif 








bool LightEnd(in LightIterator lightIt)
{
    return lightIt.m_lightCursor >= lightIt.m_lightEnd;
}





void LightNext(inout LightIterator lightIt)
{
#ifndef USE_Z_BINS
    lightIt.m_lightIndex = (texelFetch(LightListBuffer, int(++lightIt.m_lightCursor)).x & 0x7fffffffu);
#else 
    for(;;)
    {
        if(++lightIt.m_lightCursor >= lightIt.m_lightEnd)
            break;  
        lightIt.m_lightIndex = texelFetch(LightListBuffer, int(lightIt.m_lightCursor)).x;
        if((lightIt.m_lightIndex & 0x80000000u) != 0)
        {
            lightIt.m_lightIndex &= 0x7fffffffu;
            break;  
        }
        if(lightIt.m_lightIndex >= (lightIt.m_zBin >> 16) && lightIt.m_lightIndex <= (lightIt.m_zBin & 0xffffu))
            break;  
    }
#endif 
}







LightData LightGet(in LightIterator lightIt)
{
    LightData light;
    light.m_position  = texelFetch(LightBuffer, 4 * int(lightIt.m_lightIndex) + 0);
    light.m_radiance  = texelFetch(LightBuffer, 4 * int(lightIt.m_lightIndex) + 1);
    light.m_direction = texelFetch(LightBuffer, 4 * int(lightIt.m_lightIndex) + 2);
    light.m_lightData = texelFetch(LightBuffer, 4 * int(lightIt.m_lightIndex) + 3);
    return light;
}

#ifdef USE_MATERIAL_SHADER





void MaterialApplyShader(inout Material material);

#endif 








vec3 MaterialGetAlbedo(in MaterialData materialData, in vec2 uv)
{
    vec3 albedo = materialData.m_materialData1.xyz;
    uint albedoMap = floatBitsToUint(materialData.m_materialData3.x);
    if(albedoMap != uint(-1)) albedo *= Sample(albedoMap, uv).xyz;
    return albedo;
}








vec3 MaterialGetEmissivity(in MaterialData materialData, in vec2 uv)
{
    vec3 emissivity = materialData.m_materialData2.xyz;
    uint emissivityMap = floatBitsToUint(materialData.m_materialData3.y);
    if(emissivityMap != uint(-1)) emissivity *= Sample(emissivityMap, uv).xyz;
    return emissivity;
}








float MaterialGetMetallicity(in MaterialData materialData, in vec2 uv)
{
    float metallicity = materialData.m_materialData1.w;
    uint metallicityMap = floatBitsToUint(materialData.m_materialData4.z != 0.0f ? materialData.m_materialData3.w : materialData.m_materialData3.z);
    if(metallicityMap != uint(-1))
    {
        vec4 metallicityFactor = Sample(metallicityMap, uv);
        metallicity *= (materialData.m_materialData4.z != 0.0f ? metallicityFactor.z : metallicityFactor.x);
    }
    return metallicity;
}








float MaterialGetRoughness(in MaterialData materialData, in vec2 uv)
{
    float roughness = materialData.m_materialData2.w;
    uint roughnessMap = floatBitsToUint(materialData.m_materialData3.w);
    if(roughnessMap != uint(-1))
    {
        vec4 roughnessFactor = Sample(roughnessMap, uv);
        roughness *= (materialData.m_materialData4.z != 0.0f ? roughnessFactor.y : roughnessFactor.x);
    }
    return roughness;
}








vec3 MaterialGetNormal(in MaterialData materialData, in vec2 uv)
{
    uint normalMap = floatBitsToUint(materialData.m_materialData4.x);
    return (normalMap != uint(-1) ? max(Sample(normalMap, uv).xyz, 1e-3f) : vec3(0.0f));
}








float MaterialGetAo(in MaterialData materialData, in vec2 uv)
{
    uint aoMap = floatBitsToUint(materialData.m_materialData4.y);
    return (aoMap != uint(-1) ? Sample(aoMap, uv).x : 1.0f);
}







MaterialData GetMaterialData(in uint materialId)
{
    MaterialData materialData;
    if(materialId == uint(-1))
    {
        materialData.m_materialData1 = vec4(vec3(0.7f), 0.0f);
        materialData.m_materialData2 = vec4(vec3(0.0f), 0.3f);
        materialData.m_materialData3 = vec4(uintBitsToFloat(uint(-1)));
        materialData.m_materialData4 = vec4(uintBitsToFloat(uint(-1)));
    }
    else
    {
        materialData.m_materialData1 = texelFetch(MaterialBuffer, 4 * int(materialId) + 0);
        materialData.m_materialData2 = texelFetch(MaterialBuffer, 4 * int(materialId) + 1);
        materialData.m_materialData3 = texelFetch(MaterialBuffer, 4 * int(materialId) + 2);
        materialData.m_materialData4 = texelFetch(MaterialBuffer, 4 * int(materialId) + 3);
    }
    return materialData;
}












Material GetMaterialNoShade(in MaterialData materialData, in vec2 uv)
{
    Material material;
    material.m_albedo = MaterialGetAlbedo(materialData, uv);
    material.m_emissivity = MaterialGetEmissivity(materialData, uv);
    material.m_fresnelBase = vec3(0.04f);   
    material.m_normal = MaterialGetNormal(materialData, uv);
    material.m_metallicity = MaterialGetMetallicity(materialData, uv);
    material.m_roughness = MaterialGetRoughness(materialData, uv);
    material.m_ao = MaterialGetAo(materialData, uv);
    return material;
}








Material GetMaterial(in MaterialData materialData, in vec2 uv)
{
    Material material;
    material = GetMaterialNoShade(materialData, uv);
#ifdef USE_MATERIAL_SHADER
    MaterialApplyShader(material);  
    material.m_albedo = clamp(material.m_albedo, 0.0f, 1.0f);
    material.m_emissivity = max(material.m_emissivity, 0.0f);
    material.m_normal = clamp(material.m_normal, 0.0f, 1.0f);
    material.m_ao = clamp(material.m_ao, 0.0f, 1.0f);
#endif 
    material.m_albedo = pow(material.m_albedo, vec3(2.2f));
    material.m_roughness = clamp(material.m_roughness, 0.01f, 1.0f);
    material.m_metallicity = clamp(material.m_metallicity, 0.0f, 1.0f);
    material.m_fresnelBase = mix(material.m_fresnelBase, material.m_albedo, material.m_metallicity);
    material.m_roughness *= material.m_roughness;   
    return material;
}












Material GetMaterialNoShade(in uint materialId, in vec2 uv)
{
    return GetMaterialNoShade(GetMaterialData(materialId), uv);
}








Material GetMaterial(in uint materialId, in vec2 uv)
{
    return GetMaterial(GetMaterialData(materialId), uv);
}







float GetLuminance(in vec3 color)
{
    return dot(color, vec3(0.299f, 0.587f, 0.114f));
}














vec3 EvaluateNormal(in Material material, in vec3 normal, in vec2 texcoords, in vec3 viewDirection)
{
#ifndef FW_FRAGMENT_STAGE
    return normalize(normal);
#else 
    normal = normalize(normal * (gl_FrontFacing ? 1.0f : -1.0f));
    if(dot(material.m_normal, material.m_normal) == 0.0f)
        return normal;
    else
    {
        
        vec3 dp1 = dFdx(-viewDirection);
        vec3 dp2 = dFdy(-viewDirection);
        vec2 duv1 = dFdx(texcoords);
        vec2 duv2 = dFdy(texcoords);

        
        vec3 dp2perp = normalize(cross(dp2, normal));
        vec3 dp1perp = normalize(cross(normal, dp1));
        vec3 tangent = dp2perp * duv1.x + dp1perp * duv2.x;
        vec3 bitangent = dp2perp * duv1.y + dp1perp * duv2.y;

        
        float invmax = inversesqrt(max(dot(tangent, tangent), dot(bitangent, bitangent)));
        mat3 tbn = mat3(tangent * invmax, bitangent * invmax, normal);

        
        return normalize(tbn * (2.0f * material.m_normal - 1.0f));
    }
#endif 
}










float Interpolate(in float v0, in float v1, in float v2, in vec2 barycentrics)
{
    return (1.0f - barycentrics.x - barycentrics.y) * v0 + barycentrics.x * v1 + barycentrics.y * v2;
}










vec2 Interpolate(in vec2 v0, in vec2 v1, in vec2 v2, in vec2 barycentrics)
{
    return (1.0f - barycentrics.x - barycentrics.y) * v0 + barycentrics.x * v1 + barycentrics.y * v2;
}










vec3 Interpolate(in vec3 v0, in vec3 v1, in vec3 v2, in vec2 barycentrics)
{
    return (1.0f - barycentrics.x - barycentrics.y) * v0 + barycentrics.x * v1 + barycentrics.y * v2;
}










vec4 Interpolate(in vec4 v0, in vec4 v1, in vec4 v2, in vec2 barycentrics)
{
    return (1.0f - barycentrics.x - barycentrics.y) * v0 + barycentrics.x * v1 + barycentrics.y * v2;
}













float smootherstep(in float edge0, in float edge1, in float x)
{
    x = clamp((x - edge0) / (edge1 - edge0), 0.0f, 1.0f);
    return x * x * x * (x * (x * 6.0f - 15.0f) + 10.0f);
}










vec3 ViridisQuintic(in float x)
{
    x = clamp(x, 0.0f, 1.0f);

    const vec4 x1 = vec4(1.0f, x, x * x, x * x * x);    
    const vec4 x2 = x1 * x1.w * x;                      

    return vec3(
        dot(x1, vec4(+0.280268003f, -0.143510503f,  +2.225793877f, -14.815088879f)) + dot(x2.xy, vec2(+25.212752309f, -11.772589584f)),
        dot(x1, vec4(-0.002117546f, +1.617109353f,  -1.909305070f,  +2.701152864f)) + dot(x2.xy, vec2( -1.685288385f,  +0.178738871f)),
        dot(x1, vec4(+0.300805501f, +2.614650302f, -12.019139090f, +28.933559110f)) + dot(x2.xy, vec2(-33.491294770f, +13.762053843f)));
}







vec3 RGBToYCoCg(in vec3 rgb)
{
    return vec3(
         0.25f * rgb.r + 0.5f * rgb.g + 0.25f * rgb.b,
         0.5f  * rgb.r                - 0.5f  * rgb.b,
        -0.25f * rgb.r + 0.5f * rgb.g - 0.25f * rgb.b);
}







vec3 YCoCgToRGB(in vec3 yCoCg)
{
    return vec3(
        yCoCg.x + yCoCg.y - yCoCg.z,
        yCoCg.x           + yCoCg.z,
        yCoCg.x - yCoCg.y - yCoCg.z);
}





#ifdef FW_VERTEX_STAGE

layout(location = inPosition)   in vec3 Position;       
layout(location = inNormal)     in vec3 Normal;         
layout(location = inTexcoords)  in vec2 Texcoords;      
layout(location = inTexcoords2) in vec2 Texcoords2;     
layout(location = inDrawId)     in int  DrawId;         

#endif 

#endif 
