

#ifndef SHADOWS_H
#define SHADOWS_H

#include "Sampling.h"











bool GetShadowMapCoordinates(in LightData lightData, in vec2 uv, in float depth, out vec4 shadowCoords)
{
    uint shadowMapIndex = uint(-1);
    if(any(lessThanEqual(vec3(uv, depth), vec3(0.0f))) || any(greaterThanEqual(vec3(uv, depth), vec3(1.0f))))
        return false;   
    const uint lightType = floatBitsToUint(lightData.m_radiance.w),
               shadowBufferIndex = floatBitsToUint(lightData.m_lightData.z);
    if(shadowBufferIndex == uint(-1))
        return false;   
    const vec3 worldPosition = GetWorldPosition(GetScreenPosition(uv), depth);
    switch(lightType)
    {
    case lightPoint:
    case lightArea: 
        {
            const vec3 lightPosition = lightData.m_position.xyz,
                       l = worldPosition - lightPosition;
            if(abs(l.x) >= abs(l.y) && abs(l.x) >= abs(l.z))
                shadowMapIndex = (l.x > 0.0f ? 0 : 1);  
            else if(abs(l.y) >= abs(l.x) && abs(l.y) >= abs(l.z))
                shadowMapIndex = (l.y > 0.0f ? 2 : 3);  
            else
                shadowMapIndex = (l.z < 0.0f ? 4 : 5);  
        }
        break;
    case lightSpot:
        {
            shadowMapIndex = 0; 
        }
        break;
    case lightDirectional:
        {
            const int shadowSplitIndex = floatBitsToInt(lightData.m_direction.w);
            const vec4 shadowSplits = texelFetch(ShadowSplitBuffer, shadowSplitIndex);
                 if(depth < shadowSplits.x) shadowMapIndex = 0;
            else if(depth < shadowSplits.y) shadowMapIndex = 1;
            else if(depth < shadowSplits.z) shadowMapIndex = 2;
            else if(depth < shadowSplits.w) shadowMapIndex = 3;
        }
        break;
    default:
        break;  
    }
    if(shadowMapIndex == uint(-1))
        return false;   
    mat4 shadowMatrix = GetShadowMatrix(shadowBufferIndex, shadowMapIndex);
    vec4 shadowSpace = shadowMatrix * vec4(worldPosition, 1.0f);
    shadowSpace.xyz = 0.5f * shadowSpace.xyz / shadowSpace.w + 0.5f;
    shadowSpace.xyz = clamp(shadowSpace.xyz, 0.0f, 1.0f);   
    shadowCoords = vec4(shadowSpace.xyz, float(shadowMapIndex));
    return true;    
}










bool GetShadowMapCoordinates(in LightData lightData, in vec3 worldPosition, out vec4 shadowCoords)
{
    uint shadowMapIndex = uint(-1);
    const uint lightType = floatBitsToUint(lightData.m_radiance.w),
               shadowBufferIndex = floatBitsToUint(lightData.m_lightData.z);
    if(shadowBufferIndex == uint(-1))
        return false;   
    switch(lightType)
    {
    case lightPoint:
    case lightArea: 
        {
            const vec3 lightPosition = lightData.m_position.xyz,
                       l = worldPosition - lightPosition;
            if(abs(l.x) >= abs(l.y) && abs(l.x) >= abs(l.z))
                shadowMapIndex = (l.x > 0.0f ? 0 : 1);  
            else if(abs(l.y) >= abs(l.x) && abs(l.y) >= abs(l.z))
                shadowMapIndex = (l.y > 0.0f ? 2 : 3);  
            else
                shadowMapIndex = (l.z < 0.0f ? 4 : 5);  
        }
        break;
    case lightSpot:
        {
            shadowMapIndex = 0; 
        }
        break;
    case lightDirectional:
        {
            
        }
        break;
    default:
        break;  
    }
    if(shadowMapIndex == uint(-1))
        return false;   
    mat4 shadowMatrix = GetShadowMatrix(shadowBufferIndex, shadowMapIndex);
    vec4 shadowSpace = shadowMatrix * vec4(worldPosition, 1.0f);
    shadowSpace.xyz = 0.5f * shadowSpace.xyz / shadowSpace.w + 0.5f;
    shadowCoords = vec4(shadowSpace.xyz, float(shadowMapIndex));
    return true;    
}













float ResolveShadowTerm(in LightData lightData, in vec2 uv, in float depth, in float shadowBias)
{
    vec4 shadowCoords;
    float shadowTerm = 0.0f;
    const uint shadowSampleCount = 16;
    if(!GetShadowMapCoordinates(lightData, uv, depth, shadowCoords))
        return 1.0f;    
    shadowCoords.z -= shadowBias;   
    const uint shadowBufferIndex = floatBitsToUint(lightData.m_lightData.z);
    const vec2 texelSize = 1.0f / textureSize(ShadowAtlas[shadowBufferIndex], 0).xy;
    const float temporalOffset = (HasJitter() ? InterleavedGradientNoise(vec2(Frame)) : 0.0f),
                vogelAngle = 2.0f * fract(InterleavedGradientNoise(uv * ViewportWidthHeight) + temporalOffset) * FW_PI;
    for(uint i = 0; i < shadowSampleCount; ++i)
    {
        const vec2 vogelOffset = 3.0f * VogelDiskSample(i, shadowSampleCount, vogelAngle) * texelSize;
        shadowTerm += texture(ShadowAtlas[shadowBufferIndex], vec4(shadowCoords.xy + vogelOffset, shadowCoords.w, shadowCoords.z));
    }
    return shadowTerm / shadowSampleCount;
}












float ResolveShadowTerm(in LightData lightData, in vec3 worldPosition, in float shadowBias)
{
    vec4 shadowCoords;
    if(!GetShadowMapCoordinates(lightData, worldPosition, shadowCoords))
        return 1.0f;    
    shadowCoords.z -= shadowBias;   
    const uint shadowBufferIndex = floatBitsToUint(lightData.m_lightData.z);
    return texture(ShadowAtlas[shadowBufferIndex], vec4(shadowCoords.xywz));
}









float ResolveShadowTerm(in LightData lightData, in vec2 uv, in float depth)
{
    return ResolveShadowTerm(lightData, uv, depth, 0.0f);
}








float ResolveShadowTerm(in LightData lightData, in vec3 worldPosition)
{
    return ResolveShadowTerm(lightData, worldPosition, 0.0f);
}

#endif 
