

#include "ShaderCore.h"



#define DEPTH_WEIGHT    1.0f



#define PLANE_WEIGHT    1.5f



#define NORMAL_WEIGHT   1.5f

noperspective sample in vec2    vTexcoords;         
uniform sampler2D               InputBuffer;        
uniform sampler2D               DepthBuffer;        
uniform sampler2D               NormalBuffer;       
uniform sampler2D               VarianceBuffer;     
uniform vec4                    BufferDimensions;   
uniform vec2                    DenoiseAxis;        
uniform int                     DenoiseRadius;      
out vec4                        FragColor;          



struct TapKey
{
    float csZ;          
    vec3  csPosition;   
    vec3  normal;       
};







TapKey GetTapKey(in ivec2 C)
{
    TapKey key;

    if((DEPTH_WEIGHT != 0.0f) || (PLANE_WEIGHT != 0.0f))
    {
        key.csZ = GetLinearDepth(texelFetch(DepthBuffer, C, 0).x);
    }

    if(PLANE_WEIGHT != 0.0f)
    {
        key.csPosition = GetViewPosition(GetScreenPosition((C + 0.5f) * BufferDimensions.zw), texelFetch(DepthBuffer, C, 0).x);
    }

    if((NORMAL_WEIGHT != 0.0f) || (PLANE_WEIGHT != 0.0f))
    {
        key.normal = (View * vec4(normalize(2.0f * texelFetch(NormalBuffer, C, 0).xyz - 1.0f), 0.0f)).xyz;
    }

    return key;
}











float CalculateBilateralWeight(in TapKey center, in TapKey tap)
{
    float depthWeight  = 1.0f;
    float normalWeight = 1.0f;
    float planeWeight  = 1.0f;

    if(DEPTH_WEIGHT != 0.0f)
    {
        depthWeight = max(0.0f, 1.0f - abs(tap.csZ - center.csZ) * DEPTH_WEIGHT);
    }

    if(NORMAL_WEIGHT != 0.0f)
    {
        float normalCloseness = dot(tap.normal, center.normal);
        normalCloseness = normalCloseness*normalCloseness;
        normalCloseness = normalCloseness*normalCloseness;

        float normalError = (1.0f - normalCloseness);
        normalWeight = max((1.0f - normalError * NORMAL_WEIGHT), 0.0f);
    }

    if(PLANE_WEIGHT != 0.0f)
    {
        float lowDistanceThreshold2 = 0.001f;

        
        vec3 dq = center.csPosition - tap.csPosition;

        
        
        float distance2 = dot(dq, dq);

        
        float planeError = max(abs(dot(dq, tap.normal)), abs(dot(dq, center.normal)));

        planeWeight = (distance2 < lowDistanceThreshold2) ? 1.0f :
            pow(max(0.0f, 1.0f - 2.0f * PLANE_WEIGHT * planeError / sqrt(distance2)), 2.0f);
    }

    return depthWeight * normalWeight * planeWeight;
}

_kernel


void DebugInput()
{
    vec3 color = textureLod(InputBuffer, vTexcoords, 0.0f).xxx;
    FragColor = vec4(GetPassthruColor(color), 1.0f);
}

_kernel


void BilateralBlur()
{
    const ivec2 position = ivec2(gl_FragCoord.xy);
    const vec2 uv = (position + 0.5f) * BufferDimensions.zw;
    const bool isSkyPixel = (texelFetch(DepthBuffer, position, 0).x >= 1.0f);

    
    const float gaussianRadius = clamp(textureLod(VarianceBuffer, uv, 0.0f).x * 1.5f, 0.0f, 1.0f) * DenoiseRadius;

    vec3 result;

    
    if(!isSkyPixel && gaussianRadius > 0.5f)
    {
        vec3 sum = vec3(0.0f);
        float totalWeight = 0.0f;
        TapKey key = GetTapKey(position);
        const int denoiseRadius = int(gaussianRadius);

        for(int r = -denoiseRadius; r <= denoiseRadius; ++r)
        {
            const ivec2 tapOffset = ivec2(DenoiseAxis * r);
            const ivec2 tapLoc = position + tapOffset;

            const float gaussian = exp(-Square(float(r) / gaussianRadius));
            const float weight = gaussian * (r == 0 ? 1.0f : CalculateBilateralWeight(key, GetTapKey(tapLoc)));

            sum += weight * textureLod(InputBuffer, (tapLoc + 0.5f) * BufferDimensions.zw, 0.0f).xyz;
            totalWeight += weight;
        }

        
        result = sum / totalWeight;
    }
    else
    {
        
        result = textureLod(InputBuffer, (position + 0.5f) * BufferDimensions.zw, 0.0f).xyz;
    }

    
    FragColor = vec4(result, 1.0f);
}
