

#include "Sampling.h"





#define VERTEX_INDEX    ((gl_VertexID - BaseVertex) & 1)

layout(std430) buffer   AABBBuffer { AABB aabbBuffer[]; };  
layout(std430) buffer   LineBuffer { vec4 lineBuffer[]; };  
uniform vec3            WidgetPosition;                     
uniform vec3            WidgetDirection;                    
uniform float           WidgetSize;                         
uniform vec3            CrossTangent;                       
uniform vec3            CrossBitangent;                     
uniform vec3            CrossNormal;                        
uniform uint            BaseVertex;                         
uniform vec3            MinBounds;                          
uniform vec3            MaxBounds;                          
uniform mat4            Transform;                          







vec3 UnpackNormal(in float packedNormal)
{
    const uint normalUint = floatBitsToUint(packedNormal);
    const vec3 normal = vec3(
        ((normalUint >> 16) & 0xFFu) / 255.0f,
        ((normalUint >> 8 ) & 0xFFu) / 255.0f,
        ((normalUint >> 0 ) & 0xFFu) / 255.0f);
    return normalize(2.0f * normal - 1.0f);
}

_kernel


void DrawWireframeBox()
{
    float offset = Position.x;
    vec3 position = vec3(0.0f);
    switch(gl_InstanceID)
    {
    case 0:
        position = vec3(offset, -1.0f, -1.0f);
        break;
    case 1:
        position = vec3(offset,  1.0f, -1.0f);
        break;
    case 2:
        position = vec3(offset, -1.0f,  1.0f);
        break;
    case 3:
        position = vec3(offset,  1.0f,  1.0f);
        break;
    case 4:
        position = vec3(-1.0f, offset, -1.0f);
        break;
    case 5:
        position = vec3( 1.0f, offset, -1.0f);
        break;
    case 6:
        position = vec3(-1.0f, offset,  1.0f);
        break;
    case 7:
        position = vec3( 1.0f, offset,  1.0f);
        break;
    case 8:
        position = vec3(-1.0f, -1.0f, offset);
        break;
    case 9:
        position = vec3( 1.0f, -1.0f, offset);
        break;
    case 10:
        position = vec3(-1.0f,  1.0f, offset);
        break;
    case 11:
        position = vec3( 1.0f,  1.0f, offset);
        break;
    }
    vec4 worldPosition = Transform * vec4(position, 1.0f);
    gl_Position = ViewProjection * worldPosition;
}

_kernel


void DrawWireframeCross()
{
    vec3 position = Position.xxx;
    switch(gl_InstanceID)
    {
    case 0:
        position *= vec3(1.0f, 0.0f, 0.0f);
        break;
    case 1:
        position *= vec3(0.0f, 1.0f, 0.0f);
        break;
    case 2:
        position *= vec3(0.0f, 0.0f, 1.0f);
        break;
    }
    const vec3 origin = (Transform * vec4(vec3(0.0f), 1.0f)).xyz;
    const float scale = 4e-3f * dot(origin - Eye, GetForward()) / NearFar.x;
    position = ((Transform * vec4(position, 1.0f)).xyz - origin) * scale + origin;
    gl_Position = ViewProjection * vec4(position, 1.0f);
}

_kernel


void DrawWireframePlane()
{
    float offset = Position.x;
    vec3 position = vec3(0.0f);
    switch(gl_InstanceID)
    {
    case 0:
        position = vec3(offset, -1.0f, 0.0f);
        break;
    case 1:
        position = vec3(offset,  1.0f, 0.0f);
        break;
    case 2:
        position = vec3(-1.0f, offset, 0.0f);
        break;
    case 3:
        position = vec3( 1.0f, offset, 0.0f);
        break;
    }
    vec4 worldPosition = Transform * vec4(position, 1.0f);
    gl_Position = ViewProjection * worldPosition;
}

_kernel


void DrawWireframeFrustum()
{
    vec3 position = vec3(0.0f);
    switch(gl_InstanceID)
    {
    case 0:
        position = vec3(Position.x, -1.0f, -1.0f);
        break;
    case 1:
        position = vec3(Position.x,  1.0f, -1.0f);
        break;
    case 2:
        position = vec3(-1.0f, Position.x, -1.0f);
        break;
    case 3:
        position = vec3( 1.0f, Position.x, -1.0f);
        break;
    case 4:
        position = vec3(Position.x, -1.0f,  1.0f);
        break;
    case 5:
        position = vec3(Position.x,  1.0f,  1.0f);
        break;
    case 6:
        position = vec3(-1.0f, Position.x,  1.0f);
        break;
    case 7:
        position = vec3( 1.0f, Position.x,  1.0f);
        break;
    case 8:
        position = vec3(-1.0f, -1.0f, Position.x);
        break;
    case 9:
        position = vec3(-1.0f,  1.0f, Position.x);
        break;
    case 10:
        position = vec3( 1.0f, -1.0f, Position.x);
        break;
    case 11:
        position = vec3( 1.0f,  1.0f, Position.x);
        break;
    }
    position.xy *= 1.01f;   
    position.z = min(position.z, 0.65f);    
    vec4 worldPosition = Transform * vec4(position, 1.0f);
    worldPosition /= worldPosition.w;   
    gl_Position = ViewProjection * worldPosition;
}

_kernel


void DrawWireframeCircle()
{
    vec3 b1, b2;
    GetOrthoVectors(WidgetDirection, b1, b2);
    const float alpha = 2.0f * FW_PI * (gl_InstanceID + VERTEX_INDEX) / 64.0f;
    const vec3 p = (Transform * vec4(b1 * cos(alpha) + b2 * sin(alpha), 1.0f)).xyz;
    gl_Position = ViewProjection * vec4(p, 1.0f);
}

_kernel


void DrawWireframeAABB()
{
    float v = 0.5f * Position.x + 0.5f;
    vec3 position = vec3(0.0f);
    switch(gl_InstanceID)
    {
    case 0:
        position = vec3(mix(MinBounds.x, MaxBounds.x, v), MinBounds.y, MinBounds.z);
        break;
    case 1:
        position = vec3(mix(MinBounds.x, MaxBounds.x, v), MaxBounds.y, MinBounds.z);
        break;
    case 2:
        position = vec3(mix(MinBounds.x, MaxBounds.x, v), MinBounds.y, MaxBounds.z);
        break;
    case 3:
        position = vec3(mix(MinBounds.x, MaxBounds.x, v), MaxBounds.y, MaxBounds.z);
        break;
    case 4:
        position = vec3(MinBounds.x, mix(MinBounds.y, MaxBounds.y, v), MinBounds.z);
        break;
    case 5:
        position = vec3(MaxBounds.x, mix(MinBounds.y, MaxBounds.y, v), MinBounds.z);
        break;
    case 6:
        position = vec3(MinBounds.x, mix(MinBounds.y, MaxBounds.y, v), MaxBounds.z);
        break;
    case 7:
        position = vec3(MaxBounds.x, mix(MinBounds.y, MaxBounds.y, v), MaxBounds.z);
        break;
    case 8:
        position = vec3(MinBounds.x, MinBounds.y, mix(MinBounds.z, MaxBounds.z, v));
        break;
    case 9:
        position = vec3(MaxBounds.x, MinBounds.y, mix(MinBounds.z, MaxBounds.z, v));
        break;
    case 10:
        position = vec3(MinBounds.x, MaxBounds.y, mix(MinBounds.z, MaxBounds.z, v));
        break;
    case 11:
        position = vec3(MaxBounds.x, MaxBounds.y, mix(MinBounds.z, MaxBounds.z, v));
        break;
    }
    vec4 worldPosition = Transform * vec4(position, 1.0f);
    gl_Position = ViewProjection * worldPosition;
}

_kernel


void DrawWireframeAABBGPU()
{
    float v = 0.5f * Position.x + 0.5f;
    const AABB aabb = aabbBuffer[0];
    vec3 position = vec3(0.0f);
    switch(gl_InstanceID)
    {
    case 0:
        position = vec3(mix(aabb.m_minBounds.x, aabb.m_maxBounds.x, v), aabb.m_minBounds.y, aabb.m_minBounds.z);
        break;
    case 1:
        position = vec3(mix(aabb.m_minBounds.x, aabb.m_maxBounds.x, v), aabb.m_maxBounds.y, aabb.m_minBounds.z);
        break;
    case 2:
        position = vec3(mix(aabb.m_minBounds.x, aabb.m_maxBounds.x, v), aabb.m_minBounds.y, aabb.m_maxBounds.z);
        break;
    case 3:
        position = vec3(mix(aabb.m_minBounds.x, aabb.m_maxBounds.x, v), aabb.m_maxBounds.y, aabb.m_maxBounds.z);
        break;
    case 4:
        position = vec3(aabb.m_minBounds.x, mix(aabb.m_minBounds.y, aabb.m_maxBounds.y, v), aabb.m_minBounds.z);
        break;
    case 5:
        position = vec3(aabb.m_maxBounds.x, mix(aabb.m_minBounds.y, aabb.m_maxBounds.y, v), aabb.m_minBounds.z);
        break;
    case 6:
        position = vec3(aabb.m_minBounds.x, mix(aabb.m_minBounds.y, aabb.m_maxBounds.y, v), aabb.m_maxBounds.z);
        break;
    case 7:
        position = vec3(aabb.m_maxBounds.x, mix(aabb.m_minBounds.y, aabb.m_maxBounds.y, v), aabb.m_maxBounds.z);
        break;
    case 8:
        position = vec3(aabb.m_minBounds.x, aabb.m_minBounds.y, mix(aabb.m_minBounds.z, aabb.m_maxBounds.z, v));
        break;
    case 9:
        position = vec3(aabb.m_maxBounds.x, aabb.m_minBounds.y, mix(aabb.m_minBounds.z, aabb.m_maxBounds.z, v));
        break;
    case 10:
        position = vec3(aabb.m_minBounds.x, aabb.m_maxBounds.y, mix(aabb.m_minBounds.z, aabb.m_maxBounds.z, v));
        break;
    case 11:
        position = vec3(aabb.m_maxBounds.x, aabb.m_maxBounds.y, mix(aabb.m_minBounds.z, aabb.m_maxBounds.z, v));
        break;
    }
    vec4 worldPosition = Transform * vec4(position, 1.0f);
    gl_Position = ViewProjection * worldPosition;
}

_kernel


void DrawWireframeCurve()
{
    gl_Position = ViewProjection * vec4(lineBuffer[2 * gl_InstanceID + VERTEX_INDEX].xyz, 1.0f);
}

_kernel


void DrawWireframeCurveNormals()
{
    const vec4 position = lineBuffer[2 * gl_InstanceID];
    const vec3 normal = UnpackNormal(position.w);   
    const float scale = 3e-3f * dot(position.xyz - Eye, GetForward()) / NearFar.x;
    gl_Position = ViewProjection * vec4(position.xyz + scale * VERTEX_INDEX * normal, 1.0f);
}

_kernel


void DrawWireframeLineOriented()
{
    gl_Position = ViewProjection * vec4(WidgetPosition + WidgetDirection * VERTEX_INDEX, 1.0f);
}

_kernel


void DrawWireframeCrossOriented()
{
    vec3 v;
    switch(gl_InstanceID)
    {
    case 0:
        v = CrossTangent;
        break;
    case 1:
        v = CrossBitangent;
        break;
    default:
        v = CrossNormal;
        break;
    }
    float scale = 2.0f * VERTEX_INDEX - 1.0f;
    scale *= 4e-3f * dot(WidgetPosition - Eye, GetForward()) / NearFar.x;
    gl_Position = ViewProjection * vec4(WidgetPosition + v * scale, 1.0f);
}

_kernel


void DrawWireframeCircleOriented()
{
    vec3 b1, b2;
    GetOrthoVectors(WidgetDirection, b1, b2);
    const float alpha = 2.0f * FW_PI * (gl_InstanceID + VERTEX_INDEX) / 24.0f;
    gl_Position = ViewProjection * vec4(WidgetPosition + (b1 * cos(alpha) + b2 * sin(alpha)) * WidgetSize, 1.0f);
}
