const startDemoButton = document.getElementById('startDemoButton')
const canvas2D = document.getElementById('canvas-2d')
const canvasGL = document.getElementById('canvas-gl')
const canvasContainer = document.getElementById('canvasContainer')
const ctx = canvas2D.getContext('2d')
const song = document.getElementById('song')

let startTime

const gl = canvasGL.getContext('webgl') || canvasGL.getContext('experimental-webgl');

if (!gl) {
    alert('Your browser does not support WebGL.');
}

const extDrawBuffers = gl.getExtension('WEBGL_draw_buffers');
if (!extDrawBuffers) {
    alert('Your browser does not support WEBGL_draw_buffers.');
}

gl.viewport(0, 0, gl.drawingBufferWidth, gl.drawingBufferHeight);

const startDemo = () => {
    startTime = new Date().getTime() + 1*6 * 1000
    // startTime = new Date().getTime() + 0*5 * 1000 - (60/140 * (16*6 + 4)) * 1000
    startDemoButton.style.display = 'none'
    canvasContainer.style.display = 'flex'

    canvasContainer.requestFullscreen()
    .then(() => {
        requestAnimationFrame(animate)
    })
    .catch((err) => {
        alert('Failed to enter fullscreen mode:', err)
    })

    song.addEventListener('ended', () => {
        window.setTimeout(() => {
            document.exitFullscreen()
        }, (60/140 * 6) * 1000)
    })

    requestAnimationFrame(animate)
}

let songPlaying = false
const playSong = () => {
    if (songPlaying) return
    song.play()
    songPlaying = true
}

document.addEventListener("fullscreenchange", () => {
    if (!document.fullscreenElement) {
        song.pause()
    }
})

window.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        song.pause()
    }
})



// class ParameterBoxClient {
//     constructor(address) {
//       this.address = address;
//       this.ws = null;
//       this.eventListeners = { data: [] };
//     }

//     addEventListener(eventName, callback) {
//       if (!this.eventListeners[eventName]) {
//         this.eventListeners[eventName] = [];
//       }
//       this.eventListeners[eventName].push(callback);
//     }

//     _triggerEvent(eventName, ...args) {
//       if (this.eventListeners[eventName]) {
//         for (const callback of this.eventListeners[eventName]) {
//           callback(...args);
//         }
//       }
//     }

//     connect() {
//       return new Promise((resolve, reject) => {
//         this.ws = new WebSocket(this.address);

//         this.ws.addEventListener("open", () => {
//           console.log("Connection established");
//           resolve();
//         });

//         this.ws.addEventListener("close", () => {
//           console.log("Connection lost");

//           // Reconnect after 1 second
//           setTimeout(() => {
//             this.connect().catch((error) => console.error("Reconnection failed", error));
//           }, 1000);
//         });

//         this.ws.addEventListener("error", (error) => {
//           reject(error);
//         });

//         this.ws.addEventListener("message", (event) => {
//           const [maxValue, rawData] = event.data.split("; ");
//           const rawValues = rawData.split(",").map(Number);
//           const normalizedValues = rawValues.map((value) => value / parseInt(maxValue));
//           this._triggerEvent("data", normalizedValues);
//         });
//       });
//     }
//   }

// const pbox = new ParameterBoxClient("ws://localhost:8765");

// let livePboxValues = new Float32Array(16);
// pbox.addEventListener("data", (normalizedValues) => {
//     // console.log("Received normalized data:", normalizedValues);
//     for (let i = 0; i < normalizedValues.length; i++) {
//         livePboxValues[i] = normalizedValues[i];
//     }
// });

// pbox.connect().catch((error) => console.error("Connection failed", error));



const interpolateArrays = (arrayA, arrayB, t) => {
    t = 3 * Math.pow(t, 2) - 2 * Math.pow(t, 3)
    const result = new Array(arrayA.length)
    for (let i = 0; i < arrayA.length; i++) {
        result[i] = arrayA[i] * (1 - t) + arrayB[i] * t
    }
    return result
}

const beatToParam = (beat, start, end) => {
    return Math.max(0, Math.min(1, (beat - start) / (end - start)))
}


// pre-load fix
song.play()
song.pause()


const animate = () => {
    ctx.clearRect(0, 0, canvas2D.width, canvas2D.height)

    const currentTime = new Date().getTime()
    const time = (currentTime - startTime) / 1000
    const beat = time * 140 / 60 // 140bpm

    if (time < -1) {
        // t < -2 -> full opacity
        // t > -1 -> full transparency
        let opacity = Math.max(0, Math.min(1, -1-time)).toFixed(3)
        opacity = 3 * Math.pow(opacity, 2) - 2 * Math.pow(opacity, 3)

        ctx.fillStyle = `hsla(0, 0%, 10%, ${opacity})`
        ctx.fillRect(0, 0, canvas2D.width, canvas2D.height)

        ctx.font = '144px Courier New, monospace'

        ctx.fillStyle = `hsla(120, 100%, 50%, ${opacity})`

        ctx.textAlign = 'center'
        ctx.textBaseline = 'middle'
        ctx.fillText(time.toFixed(2), canvas2D.width / 2, canvas2D.height / 2)

        // preload fix
        ctx.font = '1px Londrina Solid'
        ctx.fillStyle = `hsla(0, 0%, 0%, 0.001)`
        ctx.fillText('h', canvas2D.width / 2, canvas2D.height*0.9)


    } else if (time >= 0) {
        playSong()

        canvas2D.style.display = 'none'
        canvasGL.style.display = 'block'

        // 2D Canvas

        ctx.fillStyle = 'white'
        ctx.fillRect(0, 0, canvas2D.width, canvas2D.height)

        ctx.font = '180px Londrina Solid'
        ctx.fillStyle = `black`
        ctx.textAlign = 'center'
        ctx.textBaseline = 'middle'

        if (beat-4 < 16*1) {
            ctx.font = '320px Londrina Solid'
            ctx.fillText('After', canvas2D.width / 2, canvas2D.height / 2 * 0.6)
            ctx.font = '250px Londrina Solid'
            ctx.fillText('4 long years', canvas2D.width / 2, canvas2D.height / 2 * 1.4)

        } else if (beat-4 < 16*2) {
            ctx.font = '270px Londrina Solid'
            ctx.fillText('Graffathon', canvas2D.width / 2, canvas2D.height / 2 * 0.45)
            ctx.font = '270px Londrina Solid'
            ctx.fillText('is back', canvas2D.width / 2, canvas2D.height / 2 * 1.15)
            ctx.font = '150px Londrina Solid'
            ctx.fillText('in Finland', canvas2D.width / 2, canvas2D.height / 2 * 1.75)

        } else if (beat-4 < 16*4) {
            ctx.font = '180px Londrina Solid'
            ctx.fillText('Demo hackathon', canvas2D.width / 2, canvas2D.height / 2 * 0.6)
            ctx.font = '220px Londrina Solid'
            ctx.fillText('for beginners', canvas2D.width / 2, canvas2D.height / 2 * 1.4)

        } else if (beat-4 < 16*4.75) {
            ctx.font = '300px Londrina Solid'
            ctx.fillText('Veterans', canvas2D.width / 2, canvas2D.height / 2 * 0.7)

            ctx.font = '180px Londrina Solid'
            ctx.fillText('warmly welcome', canvas2D.width / 2, canvas2D.height / 2 * 1.5)

        } else if (beat-4 < 16*5) {
            // no text

        } else if (beat-4 < 16*5.5 - 7/16) { // off-beat pad note
            ctx.font = '300px Londrina Solid'
            ctx.fillText('Challenge', canvas2D.width / 2, canvas2D.height / 2 * 0.6)
            ctx.fillText('yourself', canvas2D.width / 2, canvas2D.height / 2 * 1.4)

        } else if (beat-4 < 16*6) {
            ctx.font = '220px Londrina Solid'
            ctx.fillText('Meet new and', canvas2D.width / 2, canvas2D.height / 2 * 0.6)
            ctx.font = '250px Londrina Solid'
            ctx.fillText('old friends', canvas2D.width / 2, canvas2D.height / 2 * 1.4)

        } else if (beat-4 < 16*7) {
            ctx.font = '220px Londrina Solid'
            ctx.fillText('Enjoy', canvas2D.width / 2, canvas2D.height / 2 * 0.4)
            ctx.font = '180px Londrina Solid'
            ctx.fillText('out-of-the-box', canvas2D.width / 2, canvas2D.height / 2 * 1.0)
            ctx.font = '220px Londrina Solid'
            ctx.fillText('effects', canvas2D.width / 2, canvas2D.height / 2 * 1.6)

        } else if (beat-4 < 16*8) {
            ctx.font = '180px Londrina Solid'
            ctx.fillText('28.7. – 30.7.2023', canvas2D.width / 2, canvas2D.height / 2 * 0.7)
            ctx.fillText('Otaniemi, Finland', canvas2D.width / 2, canvas2D.height / 2 * 1.3)

        } else if (beat-4 < 16*9) {
            ctx.font = '130px Londrina Solid'
            ctx.fillText('Code + VFX by badfelix', canvas2D.width / 2, canvas2D.height / 2 * 0.7)
            ctx.fillText('Music by monisto', canvas2D.width / 2, canvas2D.height / 2 * 1.3)
        }

        updateCanvas2DTexture();

        // Shader Canvas

        let pboxValues

        const params1 = [
            0.7044,
            0.8288,
            0.4088,
            0.8224,
            0.5942,
            0.4917,
            0.0319,
            0.4850,
            0.9843,
            0.0000,
            0.1741,
            0.1642,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const params2 = [
            0.7044,
            0.8288,
            0.6773,
            0.8383,
            0.5942,
            0.4918,
            0.5636,
            0.4821,
            0.9843,
            0.0000,
            0.9913,
            0.0000,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const params3 = [
            0.5797,
            0.9445,
            0.6821,
            0.9353,
            0.5736,
            0.2597,
            0.4679,
            0.6482,
            0.8562,
            0.0000,
            0.7806,
            0.0000,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const params4 = [
            0.5797,
            0.9359,
            0.7166,
            0.9062,
            0.5736,
            0.2615,
            0.4342,
            0.7285,
            0.8562,
            0.0000,
            0.8185,
            0.0000,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsFireShower = [
            0.7364,
            0.3960,
            0.5499,
            0.4510,
            0.1391,
            0.7066,
            0.2311,
            0.5639,
            1.0000,
            1.0000,
            0.5294,
            0.3404,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsFireShower2 = [
            0.7364,
            0.3960,
            0.5499,
            0.4510,
            0.1391,
            0.7066,
            0.0000, // changed
            0.5639,
            1.0000,
            1.0000,
            0.5294,
            0.3404,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsGreets = [
            0.7364,
            0.3960,
            1.000,
            0.6835,
            0.1391,
            0.7066,
            0.3014,
            0.7827,
            1.0000,
            1.0000,
            0.4660,
            0.4782,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsSnowShower1 = [
            0.5634,
            0.8500,
            0.9938,
            0.9311,
            1.0000,
            0.7450,
            0.3507,
            0.8174,
            0.9072,
            0.7529,
            0.4635,
            0.4346,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsSnowShower2 = [
            0.5634,
            0.8500,
            0.9938,
            0.9311,
            1.0000,
            0.7100, // change this carefully
            0.3507,
            0.8174,
            0.9072,
            0.7529,
            0.4635,
            0.4346,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const snowT = Math.pow(Math.cos((beat / 2 - 0.125) * 2*Math.PI) * 0.5 + 0.5, 5.0)
        const paramsSnowShower = interpolateArrays(paramsSnowShower1, paramsSnowShower2, snowT)


        const paramsPencil = [
            0.8930, // 0.9000
            0.9318,
            0.7231,
            0.8900,
            0.0416,
            0.2723,
            0.6657,
            0.2919,
            0.2057,
            0.0000,
            0.8479,
            0.0000,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsChallenge1 = [
            0.5703,
            0.856,
            0.94,
            0.718,
            0.614,
            0.7468,
            0.7818,
            0.4909,
            1.0,
            0.0,
            0.8412,
            0.0,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsChallenge2 = [
            0.5703,
            0.856,
            0.9026,
            0.718,
            0.614,
            0.7468,
            0.7818,
            0.4909,
            1.0,
            0.0,
            0.8412,
            0.0,
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
            0.0000, // unused
        ]

        const paramsFriends1 = [0.5703, 0.856, 1.0, 0.9, 0.614, 0.7468, 0.1602, 0.62, 1.0, 0.0, 0.0, 0.0, 0.4597, 0.4537, 0.4768, 0.4742]
        const paramsFriends2 = [0.5703, 0.856, 1.0, 0.6, 0.614, 0.7468, 0.1602, 0.62, 1.0, 0.0, 0.0, 0.0, 0.4597, 0.4537, 0.4768, 0.4742]


        if (beat < 4) {
            pboxValues = params1

        } else if (beat < 13.5) {
            pboxValues = interpolateArrays(params1, params2, beatToParam(beat, 4, 13.5))

        } else if (beat < 26) {
            pboxValues = params2

        } else if (beat < 34) {
            pboxValues = interpolateArrays(params2, params1, beatToParam(beat, 26, 31))

        } else if (beat < 40) {
            pboxValues = interpolateArrays(params1, params3, beatToParam(beat, 34, 40))

        } else if (beat-4 < 16*3) {
            pboxValues = interpolateArrays(params3, params4, beatToParam(beat, 40, 44))

        } else if (beat-4 < 16*4.5) {
            pboxValues = interpolateArrays(params4, paramsSnowShower, beatToParam(beat, 50.5, 52))

        } else if (beat-4 < 16*5) {
            pboxValues = interpolateArrays(paramsSnowShower, paramsChallenge1, beatToParam(beat-4, 16*4.5, 16*5))

        } else if (beat-4 < 16*5.5 - 7/16) {
            pboxValues = interpolateArrays(paramsChallenge1, paramsChallenge2, beatToParam(beat, 84, 86))

        } else if (beat-4 < 16*6) {
            pboxValues = interpolateArrays(paramsFriends1, paramsFriends2, beatToParam(beat, 91, 99))

            // const snow = interpolateArrays(paramsSnowShower1, paramsSnowShower2, 0.12) // b from above + some
            // pboxValues = interpolateArrays(snow, paramsFireShower, beatToParam(beat, 99 , 100))

        } else if (beat-4 < 16*7 + 0.5) { // +0.5 to give sin() time to go to 1
            const t = Math.pow(Math.sin(beat * 2*Math.PI / 2) * 0.5 + 0.5, 2.0);
            pboxValues = interpolateArrays(paramsFireShower, paramsFireShower2, t)

        } else if (beat-4 < 16*8) {
            pboxValues = paramsPencil

        } else {
            pboxValues = paramsGreets
        }

        // pboxValues = paramsSnowShower


        // let colorParams = livePboxValues
        let colorParams

        const cp1 = [0.1343, 0.6194, 0.0, 0.0, 0.5486, 0.511, 0.8212, 0.7173, 0.5448, 0.9781, 0.4401, 0.3721, 0.0, 0.4539, 0.4811, 0.4619]
        const cp2 = [0.1381, 0.6194, 0.1341, 0.0, 0.6035, 0.511, 0.696, 0.7173, 0.5969, 0.7276, 0.4994, 0.4168, 0.0, 0.4686, 0.4699, 0.4633]
        const cp3 = [0.5887, 0.6193, 0.0, 0.0, 0.4655, 0.5085, 0.5024, 0.7172, 0.7537, 0.4243, 1.0, 0.3499, 0.0, 0.4289, 0.4675, 0.4437]
        const cp4 = [0.6298, 0.6193, 0.7357, 0.0, 0.4136, 0.5087, 0.1865, 0.7172, 0.7428, 0.3716, 0.7806, 0.3356, 0.0, 0.4488, 0.4981, 0.4812]
        const cp5 = [0.1022, 0.6194, 0.6596, 0.0, 0.6169, 0.5087, 0.4025, 0.7172, 0.633, 0.4828, 0.6101, 0.1782, 0.0, 0.4592, 0.4857, 0.4867]
        const cp6 = [0.1359, 0.6194, 0.1163, 0.0, 0.8065, 0.509, 0.5872, 0.7172, 0.8257, 0.5399, 0.2065, 0.6663, 0.0, 0.449, 0.4676, 0.4653] // veterans warmly welcome
        const cp7 = [0.7444, 0.6194, 0.1937, 0.0, 0.2863, 0.5086, 0.8267, 0.713, 0.8616, 0.4391, 0.5365, 0.4029, 0.0, 0.4619, 0.4788, 0.474] // challenge yourself
        const cp8 = [0.7444, 0.6194, 0.1937, 0.0, 0.2863, 0.5086, 0.8267, 0.713, 0.8616, 0.75, 0.5365, 0.75, 0.0, 0.4619, 0.4788, 0.474] // challenge yourself
        const cp9 = [0.2336, 0.6239, 1.0, 0.0, 0.5195, 0.5345, 0.7652, 0.713, 0.8344, 0.75, 0.7157, 0.75, 0.0, 0.4546, 0.4779, 0.4846] // meet friends
        const cp10 = [0.2336, 0.6239, 1.0, 0.0, 0.5195, 0.5345, 0.7652, 0.713, 0.8344, 0.4781, 0.7157, 0.4528, 0.0, 0.4546, 0.4779, 0.4846] // meet friends
        // const cp11 = [0.11, 0.624, 0.6067, 0.0, 0.9813, 0.5344, 0.6644, 0.713, 0.3428, 0.9631, 0.5445, 0.4416, 0.0, 0.4414, 0.4677, 0.4767] // enjoy effects
        const cp11 = [0.1164, 0.624, 0.6033, 0.0, 0.9738, 0.5344, 0.7274, 0.7131, 0.3301, 0.9631, 0.5446, 0.4416, 0.0, 0.4235, 0.4752, 0.4799] // enjoy effects
        // const cp12 = [0.2353, 0.1521, 0.6575, 0.0, 0.5469, 0.1706, 0.6254, 0.7131, 0.6414, 0.3461, 0.6777, 0.3956, 0.0, 0.4694, 0.4661, 0.4798] // otaniemi
        const cp12 = [0.953, 0.1521, 0.6122, 0.0, 0.15, 0.1705, 0.5333, 0.7131, 0.6998, 0.3138, 0.6506, 0.535, 0.0, 0.4729, 0.4658, 0.476] // otaniemi
        const cp13 = [0.1627, 0.1521, 0.189, 0.0, 0.9498, 0.1705, 0.2842, 0.6837, 0.3364, 0.5602, 0.9804, 0.5919, 0.0, 0.4646, 0.4769, 0.4643] // credits

        if (beat-4 < 8 - 2/4) {
            colorParams = cp1
        } else if (beat-4 < 16) {
            colorParams = interpolateArrays(cp1, cp2, beatToParam(beat-4, 8-2/4, 16)*0.6+0.4)
        } else if (beat-4 < 32) {
            colorParams = interpolateArrays(cp2, cp3, beatToParam(beat-4, 16, 20))
        } else if (beat-4 < 40) {
            colorParams = interpolateArrays(cp4, cp5, beatToParam(beat-4, 36, 40))
        } else if (beat-4 < 48) {
            colorParams = cp5
        } else if (beat-4 < 80) {
            colorParams = interpolateArrays(cp6, cp7, beatToParam(beat-4, 64, 80))
        } else if (beat-4 < 88 - 7/16) {
            colorParams = interpolateArrays(cp7, cp8, beatToParam(beat-4, 84, 88))

        } else if (beat-4 < 96) {
            colorParams = interpolateArrays(cp9, cp10, beatToParam(beat-4, 88, 92))
        } else if (beat-4 < 112) {
            colorParams = cp11
        } else if (beat-4 < 120) {
            colorParams = cp12
        } else {
            colorParams = cp13
        }


        for (let i = 0; i < 4; i++) {
            // Swap textures
            const temp = textureA;
            textureA = textureB;
            textureB = temp;

            // Swap age textures
            const ageTemp = ageTextureA;
            ageTextureA = ageTextureB;
            ageTextureB = ageTemp;

            // Set up the rendering to the framebuffer
            gl.useProgram(program);
            gl.bindFramebuffer(gl.FRAMEBUFFER, fb);
            gl.framebufferTexture2D(gl.FRAMEBUFFER, extDrawBuffers.COLOR_ATTACHMENT0_WEBGL, gl.TEXTURE_2D, textureB, 0);
            gl.framebufferTexture2D(gl.FRAMEBUFFER, extDrawBuffers.COLOR_ATTACHMENT1_WEBGL, gl.TEXTURE_2D, ageTextureB, 0);
            gl.framebufferTexture2D(gl.FRAMEBUFFER, extDrawBuffers.COLOR_ATTACHMENT2_WEBGL, gl.TEXTURE_2D, colorTexture, 0);

            // Specify the draw buffers for the framebuffer
            extDrawBuffers.drawBuffersWEBGL([
                extDrawBuffers.COLOR_ATTACHMENT0_WEBGL,
                extDrawBuffers.COLOR_ATTACHMENT1_WEBGL,
                extDrawBuffers.COLOR_ATTACHMENT2_WEBGL,
            ]);

            // Set uniforms
            const resolutionUniformLocation = gl.getUniformLocation(program, 'u_resolution');
            const textureUniformLocation = gl.getUniformLocation(program, 'u_texture');
            const timeUniformLocation = gl.getUniformLocation(program, 'u_time_beat');
            // const timeUniformLocation = gl.getUniformLocation(program, 'u_deltaT');
            const ageTextureUniformLocation = gl.getUniformLocation(program, 'u_ageTexture');
            const pboxUniformLocation = gl.getUniformLocation(program, "pbox");
            const colorsUniformLocation = gl.getUniformLocation(program, "u_color_params");

            gl.uniform2f(resolutionUniformLocation, canvasGL.width, canvasGL.height);
            gl.activeTexture(gl.TEXTURE0);
            gl.bindTexture(gl.TEXTURE_2D, textureA);
            gl.uniform1i(textureUniformLocation, 0);
            gl.uniform1f(timeUniformLocation, beat);
            gl.uniform1fv(pboxUniformLocation, pboxValues);
            gl.uniform1fv(colorsUniformLocation, colorParams);

            gl.activeTexture(gl.TEXTURE1);
            gl.bindTexture(gl.TEXTURE_2D, ageTextureA);
            gl.uniform1i(ageTextureUniformLocation, 1);

            const canvas2DTextureUniformLocation = gl.getUniformLocation(program, 'u_canvas_texture');
            gl.activeTexture(gl.TEXTURE2);
            gl.bindTexture(gl.TEXTURE_2D, canvas2DTexture);
            gl.uniform1i(canvas2DTextureUniformLocation, 2);

            // Draw to the framebuffer
            gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);
        }

    }

    // Set up the rendering to the default framebuffer (screen)
    gl.useProgram(programScreen);
    gl.bindFramebuffer(gl.FRAMEBUFFER, null);
    const colorTextureUniformLocation = gl.getUniformLocation(programScreen, 'u_colorTexture');
    gl.activeTexture(gl.TEXTURE0);
    gl.bindTexture(gl.TEXTURE_2D, colorTexture);
    gl.uniform1i(colorTextureUniformLocation, 0);

    // Draw to the screen
    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);

    requestAnimationFrame(animate)
}

startDemoButton.onclick = startDemo


// Vertex Shader
const vertexShaderSource = `
    attribute vec2 a_position;
    void main() {
        gl_Position = vec4(a_position, 0.0, 1.0);
    }
`;

// Fragment Shader
const fragmentShaderSource = `
    #extension GL_EXT_draw_buffers : require

    precision mediump float;
    uniform vec2 u_resolution;
    uniform sampler2D u_texture;
    uniform float u_time_beat;
    // uniform float u_deltaT;
    uniform float pbox[16];
    uniform float u_color_params[16];
    uniform sampler2D u_canvas_texture;

    float Pbox[16];

    // https://gist.github.com/patriciogonzalezvivo/670c22f3966e662d2f83
    vec4 permute(vec4 x){return mod(((x*34.0)+1.0)*x, 289.0);}
    vec4 taylorInvSqrt(vec4 r){return 1.79284291400159 - 0.85373472095314 * r;}

    float snoise(vec3 v){
        const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;
        const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);

        // First corner
        vec3 i  = floor(v + dot(v, C.yyy) );
        vec3 x0 =   v - i + dot(i, C.xxx) ;

        // Other corners
        vec3 g = step(x0.yzx, x0.xyz);
        vec3 l = 1.0 - g;
        vec3 i1 = min( g.xyz, l.zxy );
        vec3 i2 = max( g.xyz, l.zxy );

        //  x0 = x0 - 0. + 0.0 * C
        vec3 x1 = x0 - i1 + 1.0 * C.xxx;
        vec3 x2 = x0 - i2 + 2.0 * C.xxx;
        vec3 x3 = x0 - 1. + 3.0 * C.xxx;

        // Permutations
        i = mod(i, 289.0 );
        vec4 p = permute( permute( permute(
                    i.z + vec4(0.0, i1.z, i2.z, 1.0 ))
                + i.y + vec4(0.0, i1.y, i2.y, 1.0 ))
                + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));

        // Gradients
        // ( N*N points uniformly over a square, mapped onto an octahedron.)
        float n_ = 1.0/7.0; // N=7
        vec3  ns = n_ * D.wyz - D.xzx;

        vec4 j = p - 49.0 * floor(p * ns.z *ns.z);  //  mod(p,N*N)

        vec4 x_ = floor(j * ns.z);
        vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)

        vec4 x = x_ *ns.x + ns.yyyy;
        vec4 y = y_ *ns.x + ns.yyyy;
        vec4 h = 1.0 - abs(x) - abs(y);

        vec4 b0 = vec4( x.xy, y.xy );
        vec4 b1 = vec4( x.zw, y.zw );

        vec4 s0 = floor(b0)*2.0 + 1.0;
        vec4 s1 = floor(b1)*2.0 + 1.0;
        vec4 sh = -step(h, vec4(0.0));

        vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;
        vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;

        vec3 p0 = vec3(a0.xy,h.x);
        vec3 p1 = vec3(a0.zw,h.y);
        vec3 p2 = vec3(a1.xy,h.z);
        vec3 p3 = vec3(a1.zw,h.w);

        //Normalise gradients
        vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));
        p0 *= norm.x;
        p1 *= norm.y;
        p2 *= norm.z;
        p3 *= norm.w;

        // Mix final noise value
        vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
        m = m * m;
        return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1),
                                      dot(p2,x2), dot(p3,x3) ) );
    }

    float noise(vec3 coord) {
        float value = 0.0;
        for (int i = 0; i < 1; i++) {
            value += snoise(coord + vec3(i*1000));
            coord *= 2.0;
        }
        return value;
    }

    float perlinNoise(vec2 coord, float timeMultiplier) {
        return noise(vec3(coord * 5.0, (10.0 + u_time_beat) * 0.001 / timeMultiplier));
    }


    float a(vec4 c) {
        int aH = int(c.r * 255.0);
        int aL = int(c.g * 255.0);
        int aValue = aH * 256 + aL;
        return float(aValue) / 65535.0;
    }

    float b(vec4 c) {
        int bH = int(c.b * 255.0);
        int bL = int(c.a * 255.0);
        int bValue = bH * 256 + bL;
        return float(bValue) / 65535.0;
    }

    vec4 c(float a, float b) {
        int aValue = int(a * 65535.0);
        float r = mod(float(aValue / 256), 256.0) / 255.0;
        float g = mod(float(aValue), 256.0) / 255.0;

        int bValue = int(b * 65535.0);
        float blue= mod(float(bValue / 256), 256.0) / 255.0;
        float alpha = mod(float(bValue), 256.0) / 255.0;

        return vec4(r, g, blue, alpha);
    }

    // Laplace function for diffusion
    float laplaceA(vec2 st) {
        float sum = 0.0;
        float parameter = 0.07;
        float edge = (1.0 - parameter) * 0.25;
        float corner = parameter * 0.25;
        for (int x = -1; x <= 1; x++) {
            for (int y = -1; y <= 1; y++) {
                vec2 neighbor = mod(st + vec2(float(x), float(y)), u_resolution);
                vec2 neighborUV = neighbor / u_resolution;
                float weight = (x == 0 && y == 0) ? -1.0 : (x*x == y*y) ? corner : edge + float(y) * Pbox[9];
                sum += a(texture2D(u_texture, neighborUV)) * weight;
            }
        }
        return sum;
    }

    float laplaceB(vec2 st) {
        float sum = 0.0;
        float parameter = 0.07;
        float edge = (1.0 - parameter) * 0.25;
        float corner = parameter * 0.25;
        for (int x = -1; x <= 1; x++) {
            for (int y = -1; y <= 1; y++) {
                vec2 neighbor = mod(st + vec2(float(x), float(y)), u_resolution);
                vec2 neighborUV = neighbor / u_resolution;
                float weight = (x == 0 && y == 0) ? -1.0 : (x*x == y*y) ? corner : edge - float(y) * Pbox[9];
                sum += b(texture2D(u_texture, neighborUV)) * weight;
            }
        }
        return sum;
    }

    float reaction_A(float a, float b) {
        float growth = exp(Pbox[0] * 5.0 - 5.0);
        return a * growth - b;
    }

    float reaction_B(float a, float b) {
        float poisonGrowth = exp(Pbox[4] * 5.0 - 5.0);
        float poisonDecay = exp(Pbox[8] * 5.0 - 5.0);
        return a * poisonGrowth -b * poisonDecay;
    }

    void main() {
        vec2 coord = gl_FragCoord.xy;
        vec2 uv = coord / u_resolution;
        vec4 state = texture2D(u_texture, uv);

        for (int i = 0; i < 16; i++) {
            Pbox[i] = pbox[i];
        }

        if (texture2D(u_canvas_texture, uv).r < 0.5) {
            Pbox[0] = pbox[2];
            Pbox[1] = pbox[3];
            Pbox[4] = pbox[6];
            Pbox[5] = pbox[7];
            Pbox[8] = pbox[10];
            Pbox[9] = pbox[11];
        }

        float a = a(state);
        float b = b(state);

        // Reaction-diffusion parameters
        float diffusion_A = exp(Pbox[1] * 5.0 - 5.0);
        float diffusion_B = exp(Pbox[5] * 5.0 - 5.0);

        // Reaction-diffusion equations
        float deltaA = diffusion_A * laplaceA(coord) + reaction_A(a, b);
        float deltaB = diffusion_B * laplaceB(coord) + reaction_B(a, b);

        float u_deltaT = 3.0;
        a += u_deltaT * deltaA;
        b += u_deltaT * deltaB;

        float amp = exp(0.2606 * 6.0 - 10.0);
        a += perlinNoise(uv * 1.0, 10.0) * amp;
        a += perlinNoise(uv * 10.0, 3.16) * amp * exp(0.3497 * 4.0 - 2.0);
        a += perlinNoise(uv * 100.0, 1.0) * amp * exp(0.6897 * 4.0 - 2.0);

        a = max(0.0, min(1.0, a));
        b = max(0.0, min(1.0, b));

        gl_FragData[0] = c(a, b);


        float a_c_exp = exp((u_color_params[9] - .5) * 7.0);
        a = a * 0.95 + 0.005;
        if (a < 0.5) {
            a = pow(a * 2.0, a_c_exp) / 2.0;
        } else {
            a = 1.0 - a;
            a = pow(a * 2.0, a_c_exp) / 2.0;
            a = 1.0 - a;
        }

        float b_c_exp = exp((u_color_params[11] - .5) * 7.0);
        a = a * 0.99 + 0.005;
        if (b < 0.5) {
            b = pow(b * 2.0, b_c_exp) / 2.0;
        } else {
            b = 1.0 - b;
            b = pow(b * 2.0, b_c_exp) / 2.0;
            b = 1.0 - b;
        }

        float a_hue = u_color_params[0];
        float a_sat = exp((u_color_params[4] - .5) * 3.0);
        float a_lig = exp(-(u_color_params[8] - .5) * 7.0);
        float a_red   = pow(a*.99+.005, exp(sin(6.283 * (a_hue + 2.0/3.0)) * a_sat + a_lig));
        float a_green = pow(a*.99+.005, exp(sin(6.283 * (a_hue + 1.0/3.0)) * a_sat + a_lig));
        float a_blue  = pow(a*.99+.005, exp(sin(6.283 * (a_hue + 0.0/3.0)) * a_sat + a_lig));

        float b_hue = u_color_params[2];
        float b_sat = exp((u_color_params[6] - .5) * 3.0);
        float b_lig = exp(-(u_color_params[10] - .5) * 7.0);
        float b_red   = pow(b*.99+.005, exp(sin(6.283 * (b_hue + 2.0/3.0)) * b_sat + b_lig));
        float b_green = pow(b*.99+.005, exp(sin(6.283 * (b_hue + 1.0/3.0)) * b_sat + b_lig));
        float b_blue  = pow(b*.99+.005, exp(sin(6.283 * (b_hue + 0.0/3.0)) * b_sat + b_lig));

        float red   = 1.0 - (1.0 - a_red)   * (1.0 - b_red);
        float green = 1.0 - (1.0 - a_green) * (1.0 - b_green);
        float blue  = 1.0 - (1.0 - a_blue)  * (1.0 - b_blue);

        gl_FragData[2] = vec4(red, green, blue, 1.0);
    }
`;


// Vertex Shader for rendering to the screen
const vertexShaderScreenSource = `
    attribute vec2 a_position;
    varying vec2 v_uv;
    void main() {
        v_uv = a_position * 0.5 + 0.5;
        gl_Position = vec4(a_position, 0.0, 1.0);
    }
`;

// Fragment Shader for rendering to the screen
const fragmentShaderScreenSource = `
    precision mediump float;
    uniform sampler2D u_colorTexture;
    varying vec2 v_uv;
    void main() {
        gl_FragColor = texture2D(u_colorTexture, v_uv);
    }
`;



function createShader(gl, type, source) {
    const shader = gl.createShader(type);
    gl.shaderSource(shader, source);
    gl.compileShader(shader);

    if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
        alert('An error occurred compiling the shaders: ' + gl.getShaderInfoLog(shader));
        gl.deleteShader(shader);
        return null;
    }

    return shader;
}

const vertexShader = createShader(gl, gl.VERTEX_SHADER, vertexShaderSource);
const fragmentShader = createShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSource);

const program = gl.createProgram();
gl.attachShader(program, vertexShader);
gl.attachShader(program, fragmentShader);
gl.linkProgram(program);

if (!gl.getProgramParameter(program, gl.LINK_STATUS)) {
    alert('Unable to initialize the shader program: ' + gl.getProgramInfoLog(program));
}


const vertexShaderScreen = createShader(gl, gl.VERTEX_SHADER, vertexShaderScreenSource);
const fragmentShaderScreen = createShader(gl, gl.FRAGMENT_SHADER, fragmentShaderScreenSource);

const programScreen = gl.createProgram();
gl.attachShader(programScreen, vertexShaderScreen);
gl.attachShader(programScreen, fragmentShaderScreen);
gl.linkProgram(programScreen);

if (!gl.getProgramParameter(programScreen, gl.LINK_STATUS)) {
    alert('Unable to initialize the shader program: ' + gl.getProgramInfoLog(programScreen));
}


const positionBuffer = gl.createBuffer();
gl.bindBuffer(gl.ARRAY_BUFFER, positionBuffer);
const positions = [
    -1.0,  1.0,
    -1.0, -1.0,
     1.0,  1.0,
     1.0, -1.0,
];
gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(positions), gl.STATIC_DRAW);

const positionAttributeLocation = gl.getAttribLocation(program, 'a_position');
gl.enableVertexAttribArray(positionAttributeLocation);
gl.vertexAttribPointer(positionAttributeLocation, 2, gl.FLOAT, false, 0, 0);



// Create textures
function createTexture() {
    const texture = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texture);

    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);

    return texture;
}

let textureA = createTexture();
let textureB = createTexture();
let ageTextureA = createTexture();
let ageTextureB = createTexture();

class SeededRandom {
    constructor(seed) {
        this.x = 123456789;
        this.y = 362436069;
        this.z = 521288629;
        // seed 0 will set it as random
        this.w = seed ? seed : Math.floor(Math.random() * 0x7fffffff); // 2**31 - 1
    }

    next() {
        const t = this.x ^ (this.x << 11);
        this.x = this.y;
        this.y = this.z;
        this.z = this.w;
        this.w = this.w ^ (this.w >>> 19) ^ t ^ (t >>> 8);
        // Convert the signed 32-bit number to unsigned by using ">>> 0"
        return (this.w >>> 0) / 0xffffffff;
    }
}

const seededRand = new SeededRandom(1);

function initTexture(texture) {
    const size = canvasGL.width * canvasGL.height;
    const data = new Uint8Array(size * 4);
    for (let i = 0; i < size; i++) {
        data[i * 4] = Math.floor(seededRand.next() < 0.5 ? 0 : 255.99999);
        data[i * 4 + 1] = Math.floor(seededRand.next() < 0.5 ? 0 : 255.99999);
        data[i * 4 + 2] = Math.floor(seededRand.next() < 0.5 ? 0 : 255.99999);
        data[i * 4 + 3] = Math.floor(seededRand.next() < 0.5 ? 0 : 255.99999);
    }

    gl.bindTexture(gl.TEXTURE_2D, texture);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, canvasGL.width, canvasGL.height, 0, gl.RGBA, gl.UNSIGNED_BYTE, data);
}

// Initialize the starting state
initTexture(textureA);
initTexture(textureB);

function initAgeTexture(texture) {
    const size = canvasGL.width * canvasGL.height;
    const data = new Uint8Array(size * 4);
    for (let i = 0; i < size; i++) {
        data[i * 4] = 0;
        data[i * 4 + 1] = 0;
        data[i * 4 + 2] = 0;
        data[i * 4 + 3] = 255;
    }

    gl.bindTexture(gl.TEXTURE_2D, texture);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, canvasGL.width, canvasGL.height, 0, gl.RGBA, gl.UNSIGNED_BYTE, data);
}

initAgeTexture(ageTextureA);
initAgeTexture(ageTextureB);

// Create a new texture for color
let colorTexture = createTexture();
initTexture(colorTexture);

let canvas2DTexture = createTexture();

function updateCanvas2DTexture() {
    gl.bindTexture(gl.TEXTURE_2D, canvas2DTexture);
    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, canvas2D);
}

// Create a framebuffer
const fb = gl.createFramebuffer();