#version 460

#ifndef SPIRV_ENABLED
#extension GL_NV_gpu_shader5 : enable
#endif

#define LIGHT_PROPERTIES_BINDING 1
#ifndef MATERIAL_PROPERTIES_BINDING
#define MATERIAL_PROPERTIES_BINDING 2
#endif

#include <shaders/materials/commons.glsl >
#include <shaders/materials/commons_rand.glsl >
#include <shaders/commons_hlsl.glsl >
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/particles/particles_commons.glsl >

uniform sampler2DArray s_BlueNoise;
#include <shaders/deferred/lighting_support.glsl>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

layout(std430) buffer RibbonPositionsData {
	float position[];
} prt_position_snapshots;

struct ParticleRibbonState
{
	float generation_id;
	uint segments;
};

layout(std430) buffer RibbonStatesData {
	ParticleRibbonState states[];
} prt_states_snapshots;

struct RibbonRenderParams
{
	int   max_segments;
	int   base_idx;
	int   base_segment;
	int   particles_per_segment;
	float thickness;
	int   lights_num;
	int   material_index;
	float first_segment_time_factor;
	int   render_as_strips;
	int   _pad0;
	int   _pad1;
	int   _pad2;
};

layout(std140) uniform RibbonRenderParamsBuffer {
	RibbonRenderParams ribbon_render_params;
};

#if defined(DEFERRED_PASS) || defined(SHADOWMAP_PASS)
#ifndef SPIRV_ENABLED
out Vertex
{
	vec3 vCoords;
	vec3 vNorm;				// camera space normal
	vec3 vWorldNorm;		// world space normal
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;

out uint instanceID;
#else

layout(location = 1) out struct
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;
layout(location = 0) out uint instanceID;

#endif
#endif

vec3 hsv2rgb(vec3 c) {
	vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
	vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
	return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

vec3 rgb2hsv(vec3 c)
{
	vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
	vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
	vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));

	float d = q.x - min(q.w, q.y);
	float e = 1.0e-10;
	return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}

int calculate_vidx_for_ribbon_segment(int ribbon_id, int ribbon_segment)
{
	ribbon_segment = ribbon_render_params.base_segment - ribbon_segment;
	if (ribbon_segment < 0)
		ribbon_segment += ribbon_render_params.max_segments;
	if (ribbon_segment >= ribbon_render_params.max_segments)
		ribbon_segment -= ribbon_render_params.max_segments;
	int vidx = ribbon_segment * ribbon_render_params.particles_per_segment + ribbon_id + ribbon_render_params.base_idx;
	return vidx;
}

// should be enabled only for the transparent particles which don't land in the deferred buffer?
#ifdef DEFERRED_PASS
vec4 calculate_lighting(vec4 color, MaterialPropertiesGPU material)
{
	vec3 light_color = vec3(0.0);
	for (int light_idx = 0; light_idx < ribbon_render_params.lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];
		f16vec4 projector_color = f16vec4(0.0);
		vec3 world = vtx_output.vWorldPos;// -transform_params.vCameraPosition.xyz;

		float shadow = 0.0;
		{
			bool calculate_shadows = (light.type & LightType_Shadowcasting) != 0;
			if ((material.flags & MaterialFlag_ShadowReceive) == 0)
				calculate_shadows = false;

			if ((light.type & (LightType_Spot | LightType_Directional)) != 0 && calculate_shadows)
			{
				float penumbra_scaling_factor = 0.5;

				vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);

				float in_frustum = 0.0;
				shadow = sampleShadowPCF(
					LightShadowmapCmpSamplers[light.shadowmap_sampler0],
					vShadowCoords,
					in_frustum,
					3,
					-0.001);	// was bias = 0.0001
			}
		}

		shadow = 1.0 - shadow;

		if ((light.type & (LightType_Spot | LightType_Attenuating)) == (LightType_Spot | LightType_Attenuating))
		{
			float attenuation = light_calculate_spot_attenuation(light, world.xyz);
			shadow *= attenuation;
		}

		light_color.rgb += light.diffuse.rgb * sqrt(light.intensity) * shadow;
	}

	color.rgb *= light_color;
	color.rgb *= material.diffuse.rgb;
	color.rgb += material.emissive.rgb * material.emissive_factor;

	return color;
}
#endif

// Custom vertex modifiers params

#ifdef INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK

struct VertexInput
{
	uint id;
	vec3 pos;
	vec3 norm;
	vec4 color;
	vec2 uv0;
};

#inline <INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK>
#endif

#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id)
{
	VertexInput vtx_input;
	vtx_input.id    = ribbon_id;
	vtx_input.pos   = pos;
	vtx_input.norm  = norm;
	vtx_input.color = vec4(color, 1.0);
	vtx_input.uv0   = vec2(0.0, 0.0);

	ModifierFactor modifier_factor = modifier_factor_defaults();
	modifier_factor.factor      = 1.0;
	modifier_factor.hash        = uint(ribbon_id);
	modifier_factor.id          = uint(ribbon_id);
	modifier_factor.instance_id = uint(ribbon_id);

	modifier_factor.position    = pos;
	modifier_factor.is_spawned  = false;

	CoordinateSystemTrasforms cs_transforms;
	cs_transforms.mat_local_to_model     = transform_params.mModel;
	cs_transforms.mat_local_to_instance  = mat_identity();
	cs_transforms.mat_local_to_model_inv = transform_params.mModelInv;

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>

	pos   = vtx_input.pos;
	color = vtx_input.color.rgb;
	return pos;
}
#else
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id)
{
	return pos;
}
#endif

vec3 get_ribbon_transformed_segment_position(int ribbon_id, int segment_idx)
{
	int vidx = calculate_vidx_for_ribbon_segment(ribbon_id, segment_idx);
	vec3 p = vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);

	// TODO: Apply modifiers here!

	vec3 dummy_normal = vec3(0.0);
	vec3 dummy_color = vec3(0.0);
	vec3 modified_p = apply_vtx_modifier(p, dummy_normal, dummy_color, ribbon_id);

	return vector_transform_by_mat43(modified_p, transform_params.mModelView);
}

// ray-cylinder intersection as in https://iquilezles.org/articles/intersectors/
vec4 ray_cyl_intersect(in vec3 ro, in vec3 rd, in vec3 cb, in vec3 ca, float cr)
{
	vec3  oc = ro - cb;
	float caca = dot(ca, ca);
	float card = dot(ca, rd);
	float caoc = dot(ca, oc);
	float a = 1.0 - card * card;
	float b = dot( oc, rd) - caoc*card;
	float c = dot( oc, oc) - caoc*caoc - cr*cr;
	float h = b*b - a*c;
	if( h<0.0 ) return vec4(-1.0); //no intersection
	h = sqrt(h);

	float t = (-b - h) / a;
	float y = caoc + t * card;
	vec3  n = (oc + t * rd - ca * y / caca) / cr;

	return vec4(t, n.x, n.y, n.z);
}

void main() {

#ifdef DEFERRED_PASS
	instanceID = SYS_InstanceIndex;
#endif

	// fetch input data
	int odd_even = (SYS_VertexIndex & 1) == 0 ? 1 : -1;
	int ribbon_id;
	int ribbon_segment;

	if (ribbon_render_params.render_as_strips != 0)
	{
		ribbon_id = SYS_InstanceIndex;
		ribbon_segment = int(SYS_VertexIndex) / 2;
	}
	else
	{
		uint max_segments_uint = uint(ribbon_render_params.max_segments);
		const int RIBBONS_PER_INSTANCE = 64;
		ribbon_segment = int(SYS_VertexIndex) / 2;
		ribbon_id      = SYS_InstanceIndex * RIBBONS_PER_INSTANCE + int(uint(ribbon_segment) /max_segments_uint);
		ribbon_segment = int(uint(ribbon_segment) % max_segments_uint);
	}

	// NOTE: Here it somehow prohibts shader optimization? Or just the vs <> fs ratio is so much skewed?:(
	if (false)
	if (!prt_is_alive(ribbon_id))
	{
		gl_Position = vec4(0.0, 0.0, 0.0, -1.0);
		return;
	}

	int ribbon_segments = max(0, min(ribbon_render_params.max_segments, int(prt_states_snapshots.states[ribbon_id].segments)) - 1);
	
	ribbon_segment = min(ribbon_segment, ribbon_segments);
	int vidx = calculate_vidx_for_ribbon_segment(ribbon_id, ribbon_segment);

	vec3 vInstPosition = vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
	vec3 vInstNormal   = vec3(1.0, 0.0, 0.0);
	vec4 vInstColor    = prt_get_color(ribbon_id);

	vInstPosition      = apply_vtx_modifier(vInstPosition, vInstNormal, vInstColor.rgb, ribbon_id);

	ParticleState vState = prt_get_state(ribbon_id);

	bool is_last = ribbon_segment >= ribbon_segments;
	bool is_first = ribbon_segment == 0;

	vec3 extrusion = vec3(0.0);
	vec3 modelview_cylinder_axis = vec3(0.0, 0.0, 1.0);	// fallback

	int second_segment_offset = -1;				// by default go to previous segment
	if (ribbon_segment == 0)
		second_segment_offset = 1;				// so we can compute the axis for the cylinder. remember to reverse it

	if (true)
	{
		float shrink_factor = ribbon_render_params.first_segment_time_factor;
		float min_shrink_modifier = 0.15;

		// camera-space positions. when computing extrusions just use XY plane
		vec3 currPosition           = get_ribbon_transformed_segment_position(ribbon_id, ribbon_segment);
		vec3 nextPosition           = get_ribbon_transformed_segment_position(ribbon_id, ribbon_segment + second_segment_offset);

		// sometimes segments are too short/zero-length. try to handle this but this is kind of lost cause:(
		modelview_cylinder_axis = currPosition - nextPosition;
		if (length(modelview_cylinder_axis) <= 0.0001)
		{
			if (ribbon_segment > 1)
			{
				nextPosition = get_ribbon_transformed_segment_position(ribbon_id, ribbon_segment + second_segment_offset * 2);
				modelview_cylinder_axis = currPosition - nextPosition;
			}
		}

		modelview_cylinder_axis = normalize(modelview_cylinder_axis);
		vec2 dir2d = normalize(currPosition.xy - nextPosition.xy);	// NOTE: Don't just normalize in 3D and take XY
		dir2d = vec2(-dir2d.y, dir2d.x);
		vec3 dir = vec3(dir2d, 0.0);

		// if we are last segment then interpolate to shrink it towards previous. smoother end of trail motion
		if (ribbon_segment == ribbon_segments && ribbon_segments > 1) {
			//vInstPosition = mix(vInstPosition, nextPosition, ribbon_render_params.first_segment_time_factor);
			dir *= min_shrink_modifier;
		}

		if (is_first)
		{
			dir *= -min_shrink_modifier;
		}

		// if we are first after leading, also transition it from the single point towards full width instead of immediate smap into full width
		if (ribbon_segment == 1 && ribbon_segments > 2) {
			dir *= max(min_shrink_modifier, shrink_factor);
		}

		// if we are first before last, also transition it from the pingle point towards full width instead of immediate smap into full width
		if (ribbon_segment == (ribbon_segments - 1) && ribbon_segments > 2) {
			dir *= max(min_shrink_modifier, 1.0 - shrink_factor);
		}

		//if (ribbon_segment < ribbon_segments) {
			extrusion = dir * ribbon_render_params.thickness;
		//}

		//vInstColor.rgb = vec3((currPosition.xy - nextPosition.xy) * 0.1 * 0.5 + 0.5, 0.0);
		//vInstColor.rgb = vec3(length(currPosition.xy - nextPosition.xy));

	}
	
	// revert the xylinder axis for the first segment
	if (ribbon_segment == 0)
		modelview_cylinder_axis *= -1.0;

	if (!prt_is_alive(ribbon_id))
	{
		gl_Position = vec4(0.0, 0.0, 0.0, -1.0);
		return;
	}

	//
	// NOTE: Export main line coordinates as world coords. In addition export interpolant for the left-to-right side of the strip
	// so we can try to reconstruct proper 'tube' in ps.
	//

	vec3 pos = vInstPosition;
#ifdef DEFERRED_PASS	
	vtx_output.vLocalPos = pos;
#endif	
	vec3 vPos1 = pos;
	vec3 vPos = vector_transform_by_mat43(vPos1, transform_params.mModelView);

	// check extrusion's projection. should we limit this to some range? TODO
	if (vPos.z > 1.0)
	{
		vec4 projected_extrusion = vector_transform_by_mat_projection(vec3(extrusion.xy, vPos.z), transform_params.mProjection);
		
		projected_extrusion.xyz = projected_extrusion.xyz / projected_extrusion.w;
		float min_extrusion_size = 0.00065;		// make this resolution dependant, but should be fine for anything > 1080.0
		float projected_extrusion_size = length(projected_extrusion.xy);
		if (projected_extrusion_size > 0.0)		// should always be the case...
		{
			if (projected_extrusion_size < min_extrusion_size)
			{
				float scale = min_extrusion_size / projected_extrusion_size;
				extrusion *= scale;
				//vInstColor.rgb = vec3(1.0, 0.0, 0.0); // debug
			}
		}
	}

	vPos += extrusion * float(odd_even);

#ifdef DEFERRED_PASS
	MaterialPropertiesGPU material = materials.material_properties[ribbon_render_params.material_index];

	vtx_output.vWorldPos      = vector_transform_by_mat43(vPos1, transform_params.mModel);
	vtx_output.vWorldNorm.xyz = vec3(vInstPosition);
	vtx_output.vNorm          = vInstNormal;
	vtx_output.vCoords        = vPos.xyz;

	// In case of transparent material do the shading in VS (TODO: move to PS?)
	// Otherwise we fake the cylindrical shape by outputting position across the ribbon

#ifdef RIBBON_MATERIAL_TRANSPARENT
	vInstColor = calculate_lighting(vInstColor, material);
#else
	vtx_output.vNorm         = vec3(odd_even, 0.0, -0.5);	// slightly towards camera

	// build the cylinder in the model space. we are trying to move the extruded (in camera space) shape back into proper world space
	vec3 model_cylinder_axis = vector_transform_by_mat33(modelview_cylinder_axis, transform_params.mViewInv);
	vec3 model_on_cylinder_pos = vector_transform_by_mat43(vPos, transform_params.mViewInv);
	vec3 model_cylinder_pos  = vector_transform_by_mat43(vInstPosition, transform_params.mModel);
	vec4 cylinder_intersect  = ray_cyl_intersect(
		transform_params.vCameraPosition,
		normalize(model_on_cylinder_pos - transform_params.vCameraPosition),
		model_cylinder_pos,
		model_cylinder_axis,
		length(extrusion) * 2.0);	// NOTE: large radius for the cylinder doesn't really hurt us that much but is safer

	vtx_output.vWorldNorm       = -vec3(cylinder_intersect.yzw);
	if (cylinder_intersect.x == -1.0)
	{
		//vInstColor.rgb = vec3(1.0, 0.0, 0.0);	// debug
	}
#endif

	float life_time_factor = min(1.0, vState.life_time / vState.life_span);
	//if (life_time_factor < 0.1)
	life_time_factor = 1.0 - sin(life_time_factor * M_PI * 0.5);
	vInstColor.rgb *= 1.0 - pow(life_time_factor, 4.0);
	vtx_output.vColor = f16vec4(vInstColor);

#endif // DEFERRED_PASS

	gl_Position = vector_transform_by_mat_projection(vPos, transform_params.mProjection);
}

