#version 430

#include <shaders/materials/commons.glsl>

in vec2 vTexcoord0;
uniform sampler2D s_texture0;		// source
uniform sampler2D sImageMask;
uniform sampler2D sImageDisplacement;

// color
// output for 2 buffers

struct PostDisplaceParams
{
	vec2  resolution;
	vec2  inv_resolution;
	vec2  offset;
	vec2  radial;
	float color_separation;
	uint  included_tags;
	uint  _pad0;
	uint  _pad1;
};

layout (std140, row_major) uniform PostDisplaceParamsBuffer {
	PostDisplaceParams displace_params;
};


layout(location = 0) out vec4 outColor;

#define RADIAL_SAMPLES 16
vec3 radial(sampler2D smpl, vec3 origin_sample, vec2 position, vec2 strength)
{
    vec3 result = vec3(0.0);
    //vec2 dir = normalize(position - 0.5) / screenResolution.x;
	// just use fact it will be bigger at the edges...

	vec2 dir = position - 0.5;
    float dir_len = length(dir) - 0.2;
    if (dir_len <= 0.0)
		return origin_sample;
    dir = -strength * normalize(dir) * dir_len / displace_params.resolution.x;
#if 0
	for(int i=0; i<RADIAL_SAMPLES; i++)
	{
        vec3 smp = texture(smpl, position + i * dir).xyz;
        result = result + smp;
    }

#else
	vec2 dir_y = dir;
    vec2 dir_z = dir;
    vec2 dir_x = dir;
    for(int i=0; i<RADIAL_SAMPLES; i++)
	{
        result.x += texture(smpl, position + i * dir_x).x;
        result.y += texture(smpl, position + i * dir_y).y;
        result.z += texture(smpl, position + i * dir_z).z;
        dir_x *= 1.01;
        dir_y *= 1.02 * (1.0 + displace_params.color_separation * 0.01);
        dir_z *= 1.03 * (1.0 + displace_params.color_separation * 0.01);
    }

#endif
	
	return result / RADIAL_SAMPLES;
}


float remap( float t, float a, float b )
{
    return clamp( (t - a) / (b - a), 0.0, 1.0 );
}


vec2 remap( vec2 t, vec2 a, vec2 b )
{
    return clamp( (t - a) / (b - a), vec2(0.0), vec2(1.0) );
}

vec3 spectrum_offset_rgb( float t )
{
    float t0 = 3.0 * t - 1.5;
    vec3 ret = clamp( vec3( -t0, 1.0-abs(t0), t0), vec3(0.0), vec3(1.0));
    return ret;
}

vec3 yCgCo2rgb(vec3 ycc)
{
    float R = ycc.x - ycc.y + ycc.z;
    float G = ycc.x + ycc.y;
    float B = ycc.x - ycc.y - ycc.z;
    return vec3(R,G,B);
}

vec3 spectrum_offset_ycgco( float t )
{
    //vec3 ygo = vec3( 1.0, 1.5*t, 0.0 ); //green-pink
	//vec3 ygo = vec3( 1.0, -1.5*t, 0.0 ); //green-purple
	vec3 ygo = vec3( 1.0, 0.0, -1.25*t ); //cyan-orange
	//vec3 ygo = vec3( 1.0, 0.0, 1.5*t ); //brownyello-blue
	return yCgCo2rgb( ygo );
}

vec3 yuv2rgb( vec3 yuv )
{
	vec3 rgb;
	rgb.r = yuv.x + yuv.z * 1.13983;
	rgb.g = yuv.x + dot( vec2(-0.39465, -0.58060), yuv.yz );
	rgb.b = yuv.x + yuv.y * 2.03211;
	return rgb;
}

vec2 radialdistort(vec2 coord, vec2 amt)
{
	vec2 cc = coord - 0.5;
	return coord + 2.0 * cc * amt;
}

// Given a vec2 in [-1,+1], generate a texture coord in [0,+1]
vec2 barrelDistortion( vec2 p, vec2 amt )
{
	p = 2.0 * p - 1.0;
	/*
	const float maxBarrelPower = 5.0;
	//note: http://glsl.heroku.com/e#3290.7 , copied from Little Grasshopper
	float theta  = atan(p.y, p.x);
	vec2 radius = vec2( length(p) );
	radius = pow(radius, 1.0 + maxBarrelPower * amt);
	p.x = radius.x * cos(theta);
	p.y = radius.y * sin(theta);

	/*/
	// much faster version
	//const float maxBarrelPower = 5.0;
	//float radius = length(p);
	float maxBarrelPower = sqrt(5.0);
	float radius = dot(p,p); //faster but doesn't match above accurately
	p *= pow(vec2(radius), maxBarrelPower * amt);
	/* */
   return p * 0.5 + 0.5;
}

//note: from https://www.shadertoy.com/view/MlSXR3
vec2 brownConradyDistortion(vec2 uv, float dist)
{
	uv = uv * 2.0 - 1.0;
	// positive values of K1 give barrel distortion, negative give pincushion
	float barrelDistortion1 = 0.1 * dist; // K1 in text books
	float barrelDistortion2 = -0.025 * dist; // K2 in text books

	float r2 = dot(uv,uv);
	uv *= 1.0 + barrelDistortion1 * r2 + barrelDistortion2 * r2 * r2;
	//uv *= 1.0 + barrelDistortion1 * r2;
	
	// tangential distortion (due to off center lens elements)
	// is not modeled in this function, but if it was, the terms would go here
	return uv * 0.5 + 0.5;
}

vec2 distort( vec2 uv, float t, vec2 min_distort, vec2 max_distort )
{
	vec2 dist = mix( min_distort, max_distort, t );
	return radialdistort( uv, 2.0 * dist );
	//return barrelDistortion( uv, 1.75 * dist ); //distortion at center
	//return brownConradyDistortion( uv, 75.0 * dist.x );
}

// ====

vec3 spectrum_offset_yuv( float t )
{
	//vec3 yuv = vec3( 1.0, 3.0*t, 0.0 ); //purple-green
	//vec3 yuv = vec3( 1.0, 0.0, 2.0*t ); //purple-green
	//vec3 yuv = vec3( 1.0, -0.75*t, 0.0 ); //brownyello-blue
	vec3 yuv = vec3( 1.0, 0.0, -1.0*t ); //cyan-orange
	return yuv2rgb( yuv );
}

vec3 spectrum_offset( float t )
{
	return spectrum_offset_rgb( t );
	//return srgb2lin( spectrum_offset_rgb( t ) );
	//return lin2srgb( spectrum_offset_rgb( t ) );

	//return spectrum_offset_ycgco( t );
	//return spectrum_offset_yuv( t );
}

// ====

float nrand( vec2 n )
{
	return fract(sin(dot(n.xy, vec2(12.9898, 78.233)))* 43758.5453);
}

#define gamma 2.2

vec3 lin2srgb( vec3 c )
{
	return pow( c, vec3(gamma) );
}


vec3 srgb2lin( vec3 c )
{
	return pow( c, vec3(1.0/gamma));
}


vec3 render( vec2 uv )
{
	//return srgb2lin(texture(s_texture0, uv ).rgb );
	return texture(s_texture0, uv ).rgb;
}

float luma(vec3 v)
{
	return v.r * 0.7 + v.g * 0.2 + v.b * 0.1;
}

void main()
{
	vec4 base_color = texture(s_texture0, vTexcoord0);
#if 0
	outColor = vec4(radial(s_texture0, base_color.xyz, vTexcoord0), 1.0);
#else

	vec4 displacement_color = texture(sImageDisplacement, vTexcoord0);
	float mask_color = texture(sImageMask, vTexcoord0).r;

	if (mask_color.r == 0.0)
	{
		outColor.rgba = base_color.rgba;
		return;
	}
	//outColor.rgb = vec3(luma(displacement_color.rgb));
	//return;
	//outColor = vec4(radial(s_texture0, base_color.xyz, vTexcoord0, displace_params.radial.xy), 1.0);
	//return;

	vec2 uv = gl_FragCoord.xy/displace_params.resolution.xy;
    const float MAX_DIST_PX = 50.0;
    float max_distort_px = MAX_DIST_PX * luma(displacement_color.rgb);
    //vec2 max_distort = vec2(max_distort_px) * displace_params.radial / displace_params.resolution.xy;
    vec2 max_distort = vec2(max_distort_px) * max(displace_params.radial, displace_params.offset) * displace_params.inv_resolution.xy;
    vec2 min_distort = 0.1 * max_distort;
    //vec2 oversiz = vec2(1.0);
	vec2 oversiz = distort( vec2(1.0), 1.0, min_distort, max_distort );
	uv = remap( uv, 1.0-oversiz, oversiz );
	uv += (displace_params.offset * (displacement_color.rg)) * MAX_DIST_PX * displace_params.inv_resolution.xy;

	const int num_iter = 1;
	const float stepsiz = 1.0 / (float(num_iter)-1.0);
	float rnd = nrand( uv + fract(globals.time) );
	float t = 0.0;// rnd * stepsiz;
	
	vec3 sumcol = vec3(0.0);
	vec3 sumw = vec3(0.0);
	for ( int i=0; i<num_iter; ++i )
	{
		vec3 w = mix(spectrum_offset(t), vec3(1.0), 1.0 - displace_params.color_separation);
		sumw += w;
		vec2 uvd = distort(uv, t, min_distort, max_distort ); //TODO: move out of loop
		sumcol += w * render( uvd );
		//sumcol += w * render( uvd );
		t += stepsiz;
	}
	sumcol.rgb /= sumw;
	
	vec3 outcol = sumcol.rgb;
	//outcol = lin2srgb( outcol );
	//outcol += rnd/255.0;
	outColor = vec4( outcol, 1.0);
#endif
}
