#version 430

#include <shaders/materials/commons.glsl>

in vec2 vTexcoord0;
uniform sampler2D sImageSource;
uniform sampler2D sImageColor;
uniform sampler2D sImageAlpha;

// output for 2 buffers
layout(location = 0) out vec4 outColor;

#define OP_Replace  0
#define OP_Add      1
#define OP_Multiply 2
#define OP_Blend    3
#define OP_Negative 4

struct Image2DParams
{
	uint  op;
	uint  color_as_alpha;
	float color_scale;
	float color_base;
	float alpha_scale;
	float alpha_base;
	uint  color_repeat;
	uint  alpha_repeat;

	float color_transform_rotation;
	float alpha_transform_rotation;
	uint  color_texture_valid;
	uint  alpha_texture_valid;

	float color_aspect;
	float alpha_aspect;
	uint  color_flip_y;
	uint  alpha_flip_y;

	vec2  color_transform_translation;
	vec2  color_transform_scale;

	vec2  alpha_transform_translation;
	vec2  alpha_transform_scale;

	mat4 mat_color_rotation;
	mat4 mat_alpha_rotation;

	vec2  resolution;
	vec2  inv_resolution;

	float saturation;
};

layout (std140, row_major) uniform PostprocessParamsBuffer {
	Image2DParams image2d_params;
};

void main()
{
	vec2 flipped_texcoord = vTexcoord0;
	flipped_texcoord.y = 1.0 - flipped_texcoord.y;

	vec4 base_color = texelFetch(sImageSource, ivec2(vTexcoord0 * image2d_params.resolution), 0);
	vec2 color_translation = image2d_params.color_transform_translation;
	vec2 alpha_translation = image2d_params.alpha_transform_translation;

	#ifdef SPIRV_VULKAN
	color_translation.y = - color_translation.y;
	alpha_translation.y = - alpha_translation.y;
	#endif

	vec4 color = vec4(1.0);
	if (image2d_params.color_texture_valid != 0)
	{
		vec2 tc = vTexcoord0;
		tc = vector_transform_by_mat22((tc - 0.5) * vec2(image2d_params.color_aspect, 1.0), image2d_params.mat_color_rotation) / image2d_params.color_transform_scale / vec2(image2d_params.color_aspect, 1.0) + 0.5 + color_translation;
		if (image2d_params.color_flip_y != 0)
			tc.y = 1.0 - tc.y;

		color = textureLod(sImageColor, tc, 0.0);
	}
	
	float alpha = 1.0;
	if (image2d_params.alpha_texture_valid != 0)
	{
		vec2 tc = vTexcoord0;
		tc = vector_transform_by_mat22((tc - 0.5) * vec2(image2d_params.alpha_aspect, 1.0), image2d_params.mat_alpha_rotation) / image2d_params.alpha_transform_scale / vec2(image2d_params.alpha_aspect, 1.0) + 0.5 + alpha_translation;
		if (image2d_params.alpha_flip_y != 0)
			tc.y = 1.0 - tc.y;

		alpha = textureLod(sImageAlpha, tc, 0.0).r;
	}

	if (image2d_params.color_as_alpha != 0)
	{
		alpha = color.r * 0.7 + color.g * 0.2 + color.b * 0.1;
	}

	vec3 gray = vec3(dot(color.rgb, vec3(0.212, 0.715, 0.072)));
	color.rgb = mix(gray, color.rgb, image2d_params.saturation);

	color.rgb = color.rgb * image2d_params.color_scale + image2d_params.color_base;
	alpha     = alpha     * image2d_params.alpha_scale + image2d_params.alpha_base;

	alpha     = clamp(alpha, 0.0, 1.0);

	if (image2d_params.op == OP_Replace)
		base_color = color;
	if (image2d_params.op == OP_Add)
		base_color = base_color + color;
	if (image2d_params.op == OP_Multiply)
		base_color = base_color * color;
	if (image2d_params.op == OP_Blend)
		base_color = mix(base_color, color, vec4(alpha));
	if (image2d_params.op == OP_Negative)
	{
		base_color = mix(base_color, vec4(1.0) - base_color, vec4(color.rgb, 1.0));
	}

	base_color = max(base_color, vec4(0.0));
	outColor = base_color;
}

