#version 330 core

uniform vec2 resolution;
uniform float time;

out vec4 fragColor;

#define PI 3.14159265359
#define MAX_STEPS 1000
#define MAX_DISTANCE 100.0
#define SURFACE_DISTANCE 0.001

float hash(float n) {
    return fract(sin(n) * 43758.5453123);
}

float hash12(vec2 p) {
    return fract(sin(dot(p, vec2(127.1, 311.7))) * 43758.5453123);
}

vec3 rotate(vec3 v, float angle, vec3 axis) {
    axis = normalize(axis);
    float cosA = cos(angle);
    float sinA = sin(angle);
    return v * cosA + cross(axis, v) * sinA + axis * dot(axis, v) * (1.0 - cosA);
}

float map(vec3 p, out int objectID) {
    float floorDist = p.y;
    int floorID = 1;
    vec2 cell = floor((p.xz + 2.0) / 4.0);
    vec3 q = p;
    q.xz = mod(q.xz + 2.0, 4.0) - 2.0;
    float bounceSpeed = 0.5 * abs(sin(time));
    float bounceHeight = 1.0 + abs(sin(time));
    float yOffset = abs(sin(time * bounceSpeed)) * bounceHeight;
    q.y -= yOffset;

    float sphereDist = length(q - vec3(0.0, 1.0, 0.0)) - 1.0;
    int sphereID = 2;
    if (sphereDist < floorDist)
    {
        objectID = sphereID;
        return sphereDist;
    }
    else
    {
        objectID = floorID;
        return floorDist;
    }
}

vec3 calcNormal(vec3 p) {
    vec2 e = vec2(0.001, 0.0);
    int tmp;
    return normalize(vec3(
        map(p + e.xyy, tmp) - map(p - e.xyy, tmp),
        map(p + e.yxy, tmp) - map(p - e.yxy, tmp),
        map(p + e.yyx, tmp) - map(p - e.yyx, tmp)
    ));
}

float calcSoftShadow(vec3 ro, vec3 rd, float minT, float maxT, float k) {
    float res = 1.0;
    float t = minT;
    for (int i = 0; i < 16; i++) {
        int objectID;
        float h = map(ro + rd * t, objectID);
        if (h < 0.001) {
            return 0.0;
        }
        res = min(res, k * h / t);
        t += clamp(h, 0.01, 0.1);
        if (t > maxT) break;
    }
    return clamp(res, 0.0, 1.0);
}

float calcAO(vec3 p, vec3 n) {
    float ao = 0.0;
    float sca = 1.0;
    for (int i = 0; i < 5; i++) {
        float h = 0.02 * float(i);
        int objectID;
        float d = map(p + n * h, objectID);
        ao += (h - d) * sca;
        sca *= 0.5;
    }
    return clamp(1.0 - ao, 0.0, 1.0);
}

vec3 hsv2rgb(float h, float s, float v) {
    vec3 k = vec3(1.0, 2.0 / 3.0, 1.0 / 3.0);
    vec3 p = abs(fract(h + k.xyz) * 6.0 - 3.0);
    return v * mix(vec3(1.0), clamp(p - 1.0, 0.0, 1.0), s);
}

vec3 getColor(vec3 p, int objectID) {
    vec3 col = vec3(0.0);
    if (objectID == 2) {
        vec2 cell = floor((p.xz + 2.0) / 4.0);
        float hue = hash12(cell);
        col = hsv2rgb(hue, 0.7, 1.0);
    }
    else if (objectID == 1) {
        float checker = mod(floor(p.x) + floor(p.z), 2.0);
        col = mix(vec3(0.8), vec3(0.4), checker);
    }
    return col;
}

vec3 rayMarch(vec3 ro, vec3 rd, out vec3 hitPos, out vec3 normal, out int objectID) {
    float t = 0.0;
    objectID = -1;
    for (int i = 0; i < MAX_STEPS; i++) {
        vec3 p = ro + rd * t;
        float d = map(p, objectID);
        if (d < SURFACE_DISTANCE) {
            hitPos = p;
            normal = calcNormal(p);
            return getColor(p, objectID);
        }
        t += d;
        if (t > MAX_DISTANCE) break;
    }
    objectID = -1;
    return vec3(0.0);
}

vec3 calcLighting(vec3 p, vec3 n, vec3 rd, vec3 baseColor) {
    vec3 lightPos = vec3(0.0, 5.0, 0.0);
    vec3 lightDir = normalize(lightPos - p);
    float diff = max(dot(n, lightDir), 0.0);
    float spec = pow(max(dot(reflect(-lightDir, n), -rd), 0.0), 64.0);
    float shadow = calcSoftShadow(p + n * SURFACE_DISTANCE, lightDir, 0.05, 5.0, 8.0);
    float ao = calcAO(p, n);

    float ambient = 0.2;

    if (time > 24.5) {
        ambient = time / 26.0 ;
    }
    else if (time > 17.5) {
        ambient = 0.4;
    }

    vec3 col = baseColor * ((diff * shadow + ambient) * ao);
    col += vec3(1.0) * spec * shadow * ao;
    return col;
}

void main() {
    vec2 uv = (gl_FragCoord.xy / resolution.xy) * 2.0 - 1.0;
    uv.x *= resolution.x / resolution.y;

    float speed = 6.0;
    float amplitude = 0.3;
    float frequency = 0.2;
    float camZ = 15.0 - time * speed;
    float camX = sin(time * frequency * 2.0 * PI) * amplitude;
    float camY = 4.0;
    vec3 ro = vec3(camX, camY, camZ);

    float targetZ = camZ - 1.0;
    float targetX = sin((time + 0.5) * frequency * 2.0 * PI) * amplitude;
    float targetY = camY - 1.0;
    vec3 target = vec3(targetX, targetY, targetZ);

    vec3 forward = normalize(target - ro);
    vec3 right = normalize(cross(vec3(0.0, 1.0, 0.0), forward));
    vec3 up = cross(forward, right);
    vec3 rd = normalize(forward + uv.x * right + uv.y * up);

    vec3 finalColor = vec3(0.0);
    vec3 reflectance = vec3(1.0);

    int MAX_REFLECTIONS = 0;

    if (time > 21)
    {
	MAX_REFLECTIONS = 2;
    }
    else if (time > 14)
    {
        MAX_REFLECTIONS = 1;
    }


    for (int reflections = 0; reflections <= MAX_REFLECTIONS; reflections++) {
        vec3 hitPos, normal;
        int objectID;
        vec3 baseColor = rayMarch(ro, rd, hitPos, normal, objectID);

        if (objectID == -1) {
            vec3 skyColor = vec3(0.6, 0.8, 1.0);
            finalColor += reflectance * skyColor;
            break;
        }

        vec3 color = calcLighting(hitPos, normal, rd, baseColor);
        finalColor += reflectance * color;

        if (reflections < MAX_REFLECTIONS) {
            float reflectivity;
            if (objectID == 2) {
                reflectivity = 0.8;
            }
            else if (objectID == 1) {
                reflectivity = 0.3;
            }

            reflectance *= reflectivity;
            ro = hitPos + normal * SURFACE_DISTANCE;
            rd = reflect(rd, normal);
        }
        else {
            break;
        }
    }

    finalColor = finalColor / (1.0 + finalColor);

    finalColor = pow(finalColor, vec3(1.0 / 2.2));

    fragColor = vec4(finalColor, 1.0);
}
