#define TILE_SIZE 16

#define PI 3.14159f

#define MAT_WRITE_COLOR 0x00000001
#define MAT_READ_DEPTH	0x00000002
#define MAT_WRITE_DEPTH 0x00000004

static const float cascade_ranges[3] = {10.0f, 30.0f, 100.0f};

cbuffer PerFrame : register(b0)
{
  float4x4 g_proj;
  float4x4 g_view_proj;
  float4x4 g_inv_view_proj;
  float4x4 g_inv_proj;
  float4x4 g_view;
  float4x4 g_inv_view;
  float4x4 g_prev_view_proj;
  float4x4 g_proj_2d;

  float3 g_cam_pos;
  int g_num_dir_lights;

  float2 g_screen_size;
  int g_num_point_lights;
  int g_num_cullable_lights;

  float g_time;
  float g_delta_time;
  float2 g_jitter;

  float3 g_cam_dir;
  float g_fov;
};

struct Material
{
  float3 diffuse;
  float roughness;

  float3 specular;
  float metalness;

  float3 emissive;
  float hardness;

  int diffuse_tex;
  int normal_tex;
  int roughness_tex;
  int flags;

  float2 tiling;
  float2 custom;
};

struct Instance
{
  float4x4 world;
  float4x4 inv_world;
  int mat_index;

  float3 _pad;
};

cbuffer Postprocess : register(b0, space1)
{
  float g_exposure;
  float g_hbao_radius;
	float g_hbao_steps;
	float g_hbao_mul;

  float g_hbao_n_dot_v_bias;
  float g_bloom_mul;
  float g_dof_focal_dist;
  float g_dof_focus_range;

  float g_dof_sharp_range;
  float g_dust_flicker;
  float g_dust_size;
  float g_dust_speed;

  float4 g_dust_force;

  float g_ssr_normal_bias;
	float g_ssr_depth_bias;
  float g_ssr_stride_z_cutoff;
  float g_ssr_z_thickness;

  float g_ssr_stride;
  float g_ssr_max_steps;
  float g_fish_eye;
  float g_min_vigniette;

  float g_max_vigniette;
	float g_chromab;
	float g_min_chromab;
	float g_max_chromab;

  float3 g_dust_color;
  float g_noise_scale;

  int g_lut_idx;
  float3 g_sun_color;

  float3 g_h_color;
  float g_h_min;

  float3 g_h_color2;
  float g_h_max;

  float3 g_sun_dir;
  float g_glitch;

  float g_glitch_blend;
  float3 g_pad;
};

struct Plane
{
  float3 normal;
  float d;
};

struct Frustum
{
  Plane planes[4];
};

struct Sphere
{
  float3 pos;
  float r;
};

struct AABB
{
  float3 center;
  float3 extent;
};

Plane ComputePlane(float3 p0, float3 p1, float3 p2)
{
  Plane plane;

  float3 v0 = p1 - p0;
  float3 v1 = p2 - p0;

  plane.normal = normalize(cross(v1, v0));
  plane.d = dot( plane.normal, p0 );
  return plane;
}

float4 ClipToView( float4 clip )
{
  float4 view = mul(clip, g_inv_proj);
  view.xyz /= view.w;
  return view;
}

float4 ScreenToView( float4 screen )
{
  float2 uv = screen.xy;
  float4 clip = float4( float2(uv.x, 1.0 - uv.y) * 2.0f - 1.0f, screen.z, screen.w);

  return ClipToView(clip);
}

float3 WorldToScreen(float3 pos, float4x4 mtx)
{
  float4 clip = mul(float4(pos, 1.0f), mtx);
  clip.xyz /= clip.w;
  clip.x = clip.x * 0.5f + 0.5f;
  clip.y = -clip.y * 0.5f + 0.5f;
  return clip.xyz;
}

uint GridIndex(float2 pos_xy, float view_width)
{
  const uint2 pos = uint2(pos_xy);
  const uint tile_x = ceil(view_width / TILE_SIZE);
  return (pos.x / TILE_SIZE) + (tile_x * (pos.y / TILE_SIZE));
}

bool SphereIntersectsAABB(in Sphere sphere, in AABB aabb)
{
	float3 delta = max(0, abs(aabb.center - sphere.pos) - aabb.extent);
	float dist_sqrt = dot(delta, delta);
	return dist_sqrt <= sphere.r * sphere.r;
}

static const float ditherPattern[4][4] = {{ 0.0f, 0.5f, 0.125f, 0.625f},
{ 0.75f, 0.22f, 0.875f, 0.375f},
{ 0.1875f, 0.6875f, 0.0625f, 0.5625},
{ 0.9375f, 0.4375f, 0.8125f, 0.3125}};

static const float BayerMatrix8[8][8] =
{
	{ 1.0 / 65.0, 49.0 / 65.0, 13.0 / 65.0, 61.0 / 65.0, 4.0 / 65.0, 52.0 / 65.0, 16.0 / 65.0, 64.0 / 65.0 },
	{ 33.0 / 65.0, 17.0 / 65.0, 45.0 / 65.0, 29.0 / 65.0, 36.0 / 65.0, 20.0 / 65.0, 48.0 / 65.0, 32.0 / 65.0 },
	{ 9.0 / 65.0, 57.0 / 65.0, 5.0 / 65.0, 53.0 / 65.0, 12.0 / 65.0, 60.0 / 65.0, 8.0 / 65.0, 56.0 / 65.0 },
	{ 41.0 / 65.0, 25.0 / 65.0, 37.0 / 65.0, 21.0 / 65.0, 44.0 / 65.0, 28.0 / 65.0, 40.0 / 65.0, 24.0 / 65.0 },
	{ 3.0 / 65.0, 51.0 / 65.0, 15.0 / 65.0, 63.0 / 65.0, 2.0 / 65.0, 50.0 / 65.0, 14.0 / 65.0, 62.0 / 65.0 },
	{ 35.0 / 65.0, 19.0 / 65.0, 47.0 / 65.0, 31.0 / 65.0, 34.0 / 65.0, 18.0 / 65.0, 46.0 / 65.0, 30.0 / 65.0 },
	{ 11.0 / 65.0, 59.0 / 65.0, 7.0 / 65.0, 55.0 / 65.0, 10.0 / 65.0, 58.0 / 65.0, 6.0 / 65.0, 54.0 / 65.0 },
	{ 43.0 / 65.0, 27.0 / 65.0, 39.0 / 65.0, 23.0 / 65.0, 42.0 / 65.0, 26.0 / 65.0, 38.0 / 65.0, 22.0 / 65.0 }
};

inline float Dither(in float2 pixel)
{
	return BayerMatrix8[pixel.x % 8][pixel.y % 8];
}

float Noise(float2 co)
{
    return frac(sin(dot(co.xy ,float2(12.9898,78.233))) * 43758.5453);
}

float Luminance(float3 color)
{
  return max(0.00001f, dot(color, float3(0.2127f, 0.7152f, 0.0722f)));
}

float3 RGBtoHSV(float3 RGB)
{
    float3 HSV = 0;
    float M = min(RGB.r, min(RGB.g, RGB.b));
    HSV.z = max(RGB.r, max(RGB.g, RGB.b));
    float C = HSV.z - M;
    if (C != 0)
    {
        HSV.y = C / HSV.z;
        float3 D = (((HSV.z - RGB) / 6) + (C / 2)) / C;
        if (RGB.r == HSV.z)
            HSV.x = D.b - D.g;
        else if (RGB.g == HSV.z)
            HSV.x = (1.0/3.0) + D.r - D.b;
        else if (RGB.b == HSV.z)
            HSV.x = (2.0/3.0) + D.g - D.r;
        if ( HSV.x < 0.0 ) { HSV.x += 1.0; }
        if ( HSV.x > 1.0 ) { HSV.x -= 1.0; }
    }
    return HSV;
}
