struct DirLight
{
  float3 direction;
  float intensity;
  
  float3 color;
  float hardness;
  
  int shadow_map_index;

  float scatt_density;
  float scatt_falloff;
  float scatt_multiplier;
};

struct CullableLight
{
  float3 position;
  float radius;
  
  float3 direction;
  float cosangle;
  
  float3 color;
  float intensity;
  
  int type;
  float hardness;
  int shadow_map_index;
  float scatt_density;

  float scatt_falloff;
  float scatt_multiplier;
  float2 _pad;
};

struct CullInfo
{
  float3 position;
  float radius;
};

// https://www.unrealengine.com/en-US/blog/physically-based-shading-on-mobile
float3 EnvBRDFApprox(float3 specularColor, float roughness, float ndotv)
{
	const float4 c0 = float4(-1, -0.0275, -0.572, 0.022);
	const float4 c1 = float4(1, 0.0425, 1.04, -0.04);
	float4 r = roughness * c0 + c1;
	float a004 = min(r.x * r.x, exp2(-9.28 * ndotv)) * r.x + r.y;
	float2 AB = float2(-1.04, 1.04) * a004 + r.zw;
	return specularColor * AB.x + AB.y; 
}

//Normal distribution function
//-----------------------------------------------------------
float DistributionGGX(float n_dot_h, float roughness)
{
    float a      = roughness*roughness;
    float a2     = a*a;
    float n_dot_h_sq = n_dot_h*n_dot_h;
	
    float num   = a2;
    float denom = (n_dot_h_sq * (a2 - 1.0) + 1.0);
    denom = 3.14159f * denom * denom + 0.0000001f;
	
    return num / denom;
}

//Fresnel effect
//-----------------------------------------------------------
float3 FresnelSchlick(float v_dot_h, float3 F0)
{
    return F0 + (1.0.xxx - F0) * pow(clamp(1.0 - v_dot_h, 0.0, 1.0), 5.0);
} 

//Geometry function
//-----------------------------------------------------------	
float GeometrySchlickGGX(float n_dot_v, float roughness)
{
    float r = (roughness + 1.0);
    float k = (r*r) / 8.0;

    return n_dot_v / (n_dot_v * (1.0 - k) + k);
}

float GeometrySmith(float n_dot_v, float n_dot_l, float roughness)
{
    float ggx2  = GeometrySchlickGGX(n_dot_v, roughness);
    float ggx1  = GeometrySchlickGGX(n_dot_l, roughness);
    return ggx1 * ggx2;
}

//Cook-torrance brdf
//-----------------------------------------------------------
float3 BRDF(float3 L, float3 N, float3 V, Material mat)
{
  float3 H = normalize(L + V);
  
  float v_dot_h = saturate(dot(V,H));
  float n_dot_l = saturate(dot(N,L));
  float n_dot_h = saturate(dot(N,H));
  float l_dot_h = saturate(dot(L,H));
  float n_dot_v = saturate(dot(N,V));
  
  float roughness = clamp(mat.roughness, 0.05f, 1.0f);
  
  float3 F0 = 0.04f.xxx;
  F0 = lerp(F0, mat.specular, saturate(mat.metalness));
  float3 F = saturate(FresnelSchlick(v_dot_h, F0));

  float D = DistributionGGX(n_dot_h, roughness);
  float G = GeometrySmith(n_dot_v, n_dot_l, roughness);

  float denom = 4.0f * n_dot_v * n_dot_l + 0.001f;
  float3 specular = (D * G * F);// / denom;

  float3 kD = 1.0f.xxx - F;
  kD *= 1.0f - mat.metalness;
  
  specular += EnvBRDFApprox(mat.specular, mat.roughness, n_dot_v)*0.4f;
  
  return (kD * mat.diffuse / 3.14159f) + specular;
}

float3 calc_dir_light(DirLight light, Material mat, float3 n, float3 v, float shadow_factor)
{
  float n_dot_l = saturate(dot(light.direction,n)) * shadow_factor;
  float hardness = light.hardness * mat.hardness;
  n_dot_l = n_dot_l * hardness + 1.0 - hardness;
  
  float3 light_color = light.color * light.intensity;
  return BRDF(light.direction, n, v, mat) * n_dot_l * light_color;
}
 
float point_light_falloff(float dist, float radius)
{
  return pow(saturate(1.0f - pow(dist/radius,4.0f)),2.0f) / (pow(dist,2.0f)+1.0f);
}

float3 calc_point_light(CullableLight light, Material mat, float3 n, float3 pos, float3 v)
{
  float3 light_direction = light.position - pos;
  float light_dist = length(light_direction);
  light_direction = light_direction/light_dist;
  float falloff = point_light_falloff(light_dist, light.radius);
  float n_dot_l = saturate(dot(light_direction,n));
  float3 light_color = light.color * light.intensity * falloff;
  
  float hardness = light.hardness * mat.hardness;
  n_dot_l = n_dot_l * hardness + 1.0 - hardness;
  
  return BRDF(light_direction, n, v, mat) * n_dot_l * light_color;
}

float spot_light_falloff(CullableLight light, float3 pos)
{
  float light_dist = length(light.position - pos);
  float falloff = pow(saturate(1.0f - pow(light_dist/light.radius,4.0f)),2.0f) / (pow(light_dist,2.0f)+1.0f);
  float dir_dot_light = dot(normalize(pos - light.position), normalize(light.direction));
  float spot = clamp(dir_dot_light, light.cosangle, 1.0f);
  spot = (spot - light.cosangle) / (1.0f - light.cosangle);
  falloff *= saturate(spot);
  return falloff;
}

float3 calc_spot_light(CullableLight light, Material mat, float3 n, float3 pos, float3 v, float shadow_factor)
{
  float3 light_direction = normalize(light.position - pos);
  float falloff = spot_light_falloff(light, pos);

  float n_dot_l = saturate(dot(light_direction,n));
  float3 light_color = light.color * light.intensity * falloff;

  n_dot_l *= shadow_factor;
  
  float hardness = light.hardness * mat.hardness;
  n_dot_l = n_dot_l * hardness + 1.0 - hardness;

  return BRDF(light_direction, n, v, mat) * n_dot_l * light_color;
}