import random
import threading
import time
import json
import os
from enum import Enum
from dataclasses import dataclass
from typing import Tuple, Optional
from pynput.mouse import Controller
from pynput import keyboard
import pygetwindow as gw
import pyautogui

class PhysicsMode(Enum):
    """Different physics modes that can be applied to the mouse cursor."""
    NORMAL = "normal"       # Standard physics
    UNDERWATER = "water"    # Heavy drag, slow movement
    SPACE = "space"         # Very low gravity, low friction
    BOUNCY = "bouncy"       # Extra bouncy with high restitution
    WIND = "wind"           # Strong wind effects

@dataclass
class PhysicsConfig:
    """Configuration class for physics parameters."""
    gravity: float = 0.5
    friction: float = 0.98
    bounce_damping: float = 0.7
    wind_chance: float = 0.005
    wind_strength: float = 10.0
    mass: float = 1.0
    
    @classmethod
    def from_mode(cls, mode: PhysicsMode) -> 'PhysicsConfig':
        """Create a physics configuration from a predefined mode."""
        if mode == PhysicsMode.UNDERWATER:
            return cls(
                gravity=0.2,
                friction=0.85,
                bounce_damping=0.95,
                wind_chance=0.001,
                wind_strength=3.0,
                mass=2.0
            )
        elif mode == PhysicsMode.SPACE:
            return cls(
                gravity=0.05,
                friction=0.995,
                bounce_damping=0.9,
                wind_chance=0.0,
                wind_strength=0.0,
                mass=0.5
            )
        elif mode == PhysicsMode.BOUNCY:
            return cls(
                gravity=0.8,
                friction=0.99,
                bounce_damping=0.95,
                wind_chance=0.002,
                wind_strength=5.0,
                mass=0.8
            )
        elif mode == PhysicsMode.WIND:
            return cls(
                gravity=0.4,
                friction=0.97,
                bounce_damping=0.6,
                wind_chance=0.05,
                wind_strength=25.0,
                mass=0.7
            )
        else:  # Default NORMAL mode
            return cls()
    
    def to_dict(self) -> dict:
        """Convert config to dictionary for serialization."""
        return {
            "gravity": self.gravity,
            "friction": self.friction,
            "bounce_damping": self.bounce_damping,
            "wind_chance": self.wind_chance,
            "wind_strength": self.wind_strength,
            "mass": self.mass
        }
    
    @classmethod
    def from_dict(cls, data: dict) -> 'PhysicsConfig':
        """Create config from dictionary."""
        return cls(
            gravity=data.get("gravity", 0.5),
            friction=data.get("friction", 0.98),
            bounce_damping=data.get("bounce_damping", 0.7),
            wind_chance=data.get("wind_chance", 0.005),
            wind_strength=data.get("wind_strength", 10.0),
            mass=data.get("mass", 1.0)
        )


class GravityMouse:
    """Main class for the Gravity Mouse application."""
    def __init__(self):
        # Settings file path
        self.settings_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), "gravity_mouse_settings.json")
        
        # Mouse and screen setup
        self.mouse = Controller()
        self.screen_width, self.screen_height = pyautogui.size()
        
        # Load or initialize configuration
        self.config = self._load_config()
        
        # Physics state
        self.position_x, self.position_y = self.mouse.position
        self.velocity_x = 0.0
        self.velocity_y = 0.0
        self.last_update_time = time.time()
        self.physics_active = True
        self.current_mode = PhysicsMode.NORMAL
        
        # Alt key state tracking
        self.alt_pressed = False
        
        # Tracking state
        self.running = False
        self.physics_thread = None
        self.keyboard_listener = None
    
    def _load_config(self) -> PhysicsConfig:
        """Load configuration from file or use defaults."""
        try:
            if os.path.exists(self.settings_file):
                with open(self.settings_file, 'r') as f:
                    data = json.load(f)
                return PhysicsConfig.from_dict(data)
        except Exception as e:
            print(f"Error loading settings: {e}")
        return PhysicsConfig()
    
    def _save_config(self):
        """Save current configuration to file."""
        try:
            with open(self.settings_file, 'w') as f:
                json.dump(self.config.to_dict(), f, indent=2)
        except Exception as e:
            print(f"Error saving settings: {e}")
    
    def _on_key_press(self, key):
        """Handle keyboard input."""
        try:
            # Track Alt key
            if key == keyboard.Key.alt:
                self.alt_pressed = True
                return
                
            # Check for ALT+ESC combo
            if self.alt_pressed and key == keyboard.Key.esc:
                # Toggle physics
                self.physics_active = not self.physics_active
                state = "ON" if self.physics_active else "OFF"
                print(f"Physics: {state}")
                return
            
            # Mode switching
            if hasattr(key, 'char'):
                if key.char == '1':
                    self._change_mode(PhysicsMode.NORMAL)
                elif key.char == '2':
                    self._change_mode(PhysicsMode.UNDERWATER)
                elif key.char == '3':
                    self._change_mode(PhysicsMode.SPACE)
                elif key.char == '4':
                    self._change_mode(PhysicsMode.BOUNCY)
                elif key.char == '5':
                    self._change_mode(PhysicsMode.WIND)
                elif key.char == 's':
                    # Save current configuration
                    self._save_config()
                    print("Settings saved")
        except AttributeError:
            pass
    
    def _on_key_release(self, key):
        """Handle keyboard release events."""
        if key == keyboard.Key.alt:
            self.alt_pressed = False
    
    def _change_mode(self, mode: PhysicsMode):
        """Change the current physics mode."""
        self.current_mode = mode
        self.config = PhysicsConfig.from_mode(mode)
        print(f"Physics Mode: {mode.value}")
    
    def _physics_loop(self):
        """Main physics simulation loop."""
        last_mouse_pos = self.mouse.position
        
        while self.running:
            # Calculate delta time for smoother physics
            current_time = time.time()
            dt = min(current_time - self.last_update_time, 0.05)  # Cap at 50ms to prevent huge jumps
            self.last_update_time = current_time
            
            # Sleep to maintain frame rate
            time.sleep(max(0.01 - dt, 0))  # Target ~100 FPS but adjust based on actual frame time
            
            # Skip physics if disabled
            if not self.physics_active:
                self.position_x, self.position_y = self.mouse.position
                self.velocity_x = 0
                self.velocity_y = 0
                continue
            
            # Get current mouse position
            current_mouse_pos = self.mouse.position
            
            # If the user moved the mouse, update position but keep velocity
            if current_mouse_pos != (int(self.position_x), int(self.position_y)):
                # Calculate new velocity based on user movement
                move_x = current_mouse_pos[0] - last_mouse_pos[0]
                move_y = current_mouse_pos[1] - last_mouse_pos[1]
                
                # Only update if significant movement detected (not physics-induced)
                if abs(move_x) > 0 or abs(move_y) > 0:
                    self.position_x, self.position_y = current_mouse_pos
                    # Set some initial velocity based on user's movement
                    self.velocity_x = move_x * 0.5
                    self.velocity_y = move_y * 0.5
            
            last_mouse_pos = current_mouse_pos
            
            # Apply forces
            
            # Gravity (adjusted by mass)
            self.velocity_y += self.config.gravity / self.config.mass
            
            # Random wind gusts
            if random.random() < self.config.wind_chance:
                gust = random.uniform(-self.config.wind_strength, self.config.wind_strength)
                self.velocity_x += gust / self.config.mass
            
            # Apply friction (air resistance)
            self.velocity_x *= self.config.friction
            self.velocity_y *= self.config.friction
            
            # Update position
            self.position_x += self.velocity_x
            self.position_y += self.velocity_y
            
            # Bounce off edges with improved physics
            if self.position_x <= 0 or self.position_x >= self.screen_width:
                # Constrain position
                self.position_x = max(0, min(self.position_x, self.screen_width))
                # Reverse velocity with damping
                self.velocity_x = -self.velocity_x * self.config.bounce_damping
            
            if self.position_y <= 0 or self.position_y >= self.screen_height:
                # Constrain position
                self.position_y = max(0, min(self.position_y, self.screen_height))
                # Reverse velocity with damping
                self.velocity_y = -self.velocity_y * self.config.bounce_damping
            
            # Set the mouse position
            self.mouse.position = (int(self.position_x), int(self.position_y))
    
    def start(self):
        """Start the gravity mouse application."""
        if self.running:
            print("Gravity Mouse is already running!")
            return
            
        self.running = True
        
        # Start physics thread
        self.physics_thread = threading.Thread(target=self._physics_loop)
        self.physics_thread.daemon = True
        self.physics_thread.start()
        
        # Start keyboard listener
        self.keyboard_listener = keyboard.Listener(
            on_press=self._on_key_press,
            on_release=self._on_key_release
        )
        self.keyboard_listener.start()
        
        print("\n=== Gravity Mouse Enhanced ===")
        print("Your mouse now obeys the laws of physics!")
        print("\nControls:")
        print("  ALT+ESC: Toggle physics ON/OFF")
        print("  1-5: Switch physics modes")
        print("    1: Normal")
        print("    2: Underwater")
        print("    3: Space")
        print("    4: Bouncy")
        print("    5: Wind")
        print("  S: Save current settings")
        
    def stop(self):
        """Stop the gravity mouse application."""
        self.running = False
        
        if self.keyboard_listener:
            self.keyboard_listener.stop()
            
        if self.physics_thread:
            self.physics_thread.join(timeout=1.0)
            
        print("Gravity Mouse stopped.")
        
def main():
    """Main entry point for the application."""
    gravity_mouse = GravityMouse()
    gravity_mouse.start()
    
    try:
        # Keep main thread alive
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        gravity_mouse.stop()
        print("Exited gracefully.")

if __name__ == '__main__':
    main()
