#version 460

#define LIGHT_PROPERTIES_BINDING 1
#ifndef MATERIAL_PROPERTIES_BINDING
#define MATERIAL_PROPERTIES_BINDING 2
#endif

#include <shaders/materials/commons.glsl>
#include <shaders/materials/commons_rand.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/particles/particles_commons.glsl>

uniform sampler2DArray s_BlueNoise;
uniform sampler2D      s_NoiseRGBA;

#include <shaders/deferred/lighting/lighting_support.glsl>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

layout(std430) buffer readonly restrict RibbonPositionsData {
	float position[];
} prt_position_snapshots;

struct ParticleRibbonState
{
	float generation_id;
	uint segment_info;
};

struct RibbonRenderParams
{
	int   max_segments;
	int   base_idx_;
	int   base_segment_;
	int   particles_per_segment;
	float thickness;
	int   lights_num;
	int   material_index;
	float first_segment_time_factor;
	int   render_as_strips;
	int   segment_subdivisions;
	int   export_ribbon_t_as_modifier_factor;
	int   export_segment_t_as_modifier_factor;
};

layout(std140) uniform RibbonRenderParamsBuffer {
	RibbonRenderParams ribbon_render_params;
};


struct ParticleRibbonSegmentInfo
{
	uint segments;	// 0...max_segments-1
	uint first;		// first is always in range, rolling!
	uint previous;	// not stored, just for convenience
};

ParticleRibbonSegmentInfo ribbon_decode_segment_info(uint segment_info)
{
	ParticleRibbonSegmentInfo si;
	si.segments = segment_info & 0xffff;
	si.first = segment_info >> 16;
	si.previous = si.first - 1;
	if (si.first == 0)
		si.previous = ribbon_render_params.max_segments - 1;
	return si;
}

layout(std430) buffer readonly restrict RibbonStatesData {
	ParticleRibbonState states[];
} prt_states_snapshots;

#if defined(DEFERRED_PASS) || defined(SHADOWMAP_PASS)

layout(location = 1) out struct
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;
layout(location = 0) out uint instanceID;

#endif

int calculate_vidx_for_ribbon_segment(uint ribbon_id, int ribbon_segment)
{
	ParticleRibbonSegmentInfo ribbon_si = ribbon_decode_segment_info(prt_states_snapshots.states[ribbon_id].segment_info);
	
	ribbon_segment = int(ribbon_si.first) - ribbon_segment;
	if (ribbon_segment < 0)
		ribbon_segment += ribbon_render_params.max_segments;

	int vidx = ribbon_segment * ribbon_render_params.particles_per_segment + int(ribbon_id);
	return vidx;
}

uint get_ribbon_segments(uint ribbon_id)
{
	ParticleRibbonSegmentInfo ribbon_si = ribbon_decode_segment_info(prt_states_snapshots.states[ribbon_id].segment_info);
	int s = max(0, min(ribbon_render_params.max_segments, int(ribbon_si.segments)) - 1);
	return max(1, s);
}

vec3 ribbon_get_coords_for_vidx(int vidx)
{
	return vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
}

vec3 ribbon_get_coords_for_segment(uint ribbon_id, int segment_idx)
{
	uint vidx = calculate_vidx_for_ribbon_segment(ribbon_id, segment_idx);
	return vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
}

vec3 ribbon_get_coords_for_t(uint ribbon_idx, uint segment_idx, float t)
{
	int control_points_num = int(get_ribbon_segments(ribbon_idx)) + 1;
	int s0 = int(segment_idx);
	int s1 = min(s0 + 1, control_points_num - 1); // might be same as s0

	int vidx0 = calculate_vidx_for_ribbon_segment(ribbon_idx, s0);
	int vidx1 = calculate_vidx_for_ribbon_segment(ribbon_idx, s1);
	vec3 p0 = ribbon_get_coords_for_vidx(vidx0);
	vec3 p1 = ribbon_get_coords_for_vidx(vidx1);

	return mix(p0, p1, t);
}

vec3 catmull_rom(vec3 p0, vec3 p1, vec3 p2, vec3 p3, float t)
{
	float t2 = t * t;
	float t3 = t2 * t;

	return 0.5 * (
		(2.0 * p1) +
		(-p0 + p2) * t +
		(2.0 * p0 - 5.0 * p1 + 4.0 * p2 - p3) * t2 +
		(-p0 + 3.0 * p1 - 3.0 * p2 + p3) * t3
	);
}

vec3 b_spline(vec3 p0, vec3 p1, vec3 p2, vec3 p3, float t)
{
	float t2 = t * t;
	float t3 = t2 * t;

	vec4 T = vec4(t3, t2, t, 1.0);

	mat4 B = (1.0 / 6.0) * mat4(
        -1.0, 3.0, -3.0, 1.0,
         3.0, -6.0, 0.0, 4.0,
        -3.0, 3.0, 3.0, 1.0,
         1.0, 0.0, 0.0, 0.0
    );

	vec4 coeffs = T * B;

	return coeffs.x * p0 +
           coeffs.y * p1 +
           coeffs.z * p2 +
           coeffs.w * p3;
}

vec3 ribbon_get_coords_for_t_spline(uint ribbon_idx, uint segment_idx, float t)
{
	//return ribbon_get_coords_for_t(ribbon_idx, segment_idx, t);

	int control_points_num = int(get_ribbon_segments(ribbon_idx)) + 1;
	int s1 = int(segment_idx);
	int s2 = min(s1 + 1, control_points_num - 1); // might be same as s0

	int s0 = max(s1 - 1, 0);
	int s3 = min(s2 + 1, control_points_num - 1);

	int vidx0 = calculate_vidx_for_ribbon_segment(ribbon_idx, s0);
	int vidx1 = calculate_vidx_for_ribbon_segment(ribbon_idx, s1);
	int vidx2 = calculate_vidx_for_ribbon_segment(ribbon_idx, s2);
	int vidx3 = calculate_vidx_for_ribbon_segment(ribbon_idx, s3);
	vec3 p0 = ribbon_get_coords_for_vidx(vidx0);
	vec3 p1 = ribbon_get_coords_for_vidx(vidx1);
	vec3 p2 = ribbon_get_coords_for_vidx(vidx2);
	vec3 p3 = ribbon_get_coords_for_vidx(vidx3);

	if (s0 == s1)	// no previous point
		p0 = p1 - (p2 - p1);
	if (s3 == s2)	// no next after second point
		p3 = p2 + (p2 - p1);

	// this is ugly attempt to 'normalize' the spline
	if (false)
	{
		float d = length(p1 - p0);
		p0 = p1 - normalize(p1 - p0) * d;
		p3 = p2 + normalize(p3 - p2) * d;
	}

	return catmull_rom(p0, p1, p2, p3, t);
	//return b_spline(p0, p1, p2, p3, t);
	
	vec3 a0 = p3 - p2 - p0 + p1;
	vec3 a1 = p0 - p1 - a0;
	vec3 a2 = p2 - p0;
	vec3 a3 = p1;

	float t2 = t * t;
	return a0 * t * t2 + a1 * t2 + a2 * t + a3;
}

float compute_t_from_subsegment(int sub_segment_idx)
{
	return float(sub_segment_idx) / float(ribbon_render_params.segment_subdivisions);
}

vec3 ribbon_get_coords_for_segment_t(uint ribbon_id, int segment_idx, float t)
{
	return ribbon_get_coords_for_t_spline(ribbon_id, segment_idx, t);
}

// should be enabled only for the transparent particles which don't land in the deferred buffer?
#ifdef DEFERRED_PASS
vec4 calculate_lighting(vec4 color, MaterialPropertiesGPU material)
{
	vec3 light_color = vec3(0.0);
	for (int light_idx = 0; light_idx < ribbon_render_params.lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];
		f16vec4 projector_color = f16vec4(0.0);
		vec3 world = vtx_output.vWorldPos;// -transform_params.vCameraPosition.xyz;

		float shadow = 0.0;
		{
			bool calculate_shadows = (light.type & LightType_Shadowcasting) != 0;
			if ((material.flags & MaterialFlag_ShadowReceive) == 0)
				calculate_shadows = false;

			if ((light.type & (LightType_Spot | LightType_Directional)) != 0 && calculate_shadows)
			{
				float penumbra_scaling_factor = 0.5;

				vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);

				float in_frustum = 0.0;
				shadow = sampleShadowPCF(
					LightShadowmapCmpSamplers[light.shadowmap_sampler0],
					vShadowCoords,
					in_frustum,
					3,
					-0.001);	// was bias = 0.0001
			}
		}

		shadow = 1.0 - shadow;

		if ((light.type & (LightType_Spot | LightType_Attenuating)) == (LightType_Spot | LightType_Attenuating))
		{
			float attenuation = light_calculate_spot_attenuation(light, world.xyz);
			shadow *= attenuation;
		}

		light_color.rgb += light.diffuse.rgb * sqrt(light.intensity) * shadow;
	}

	color.rgb *= light_color;
	color.rgb *= material.diffuse.rgb;
	color.rgb += material.emissive.rgb * material.emissive_factor;

	return color;
}
#endif

// Custom vertex modifiers params

#ifdef INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK

struct VertexInput
{
	uint id;
	vec3 pos;
	vec3 norm;
	vec4 color;
	vec2 uv0;
};

#inline <INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK>
#endif

#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id, float t)
{
	VertexInput vtx_input;
	vtx_input.id    = ribbon_id;
	vtx_input.pos   = pos;
	vtx_input.norm  = norm;
	vtx_input.color = vec4(color, 1.0);
	vtx_input.uv0   = vec2(0.0, 0.0);

	ModifierFactor modifier_factor = modifier_factor_defaults();
	modifier_factor.factor      = t;
	modifier_factor.hash        = uint(ribbon_id);
	modifier_factor.id          = uint(ribbon_id);
	modifier_factor.instance_id = uint(ribbon_id);

	modifier_factor.position    = pos;
	modifier_factor.is_spawned  = false;

	CoordinateSystemTrasforms cs_transforms;
	cs_transforms.mat_local_to_model     = transform_params.mModel;
	cs_transforms.mat_local_to_instance  = mat_identity();
	cs_transforms.mat_local_to_model_inv = transform_params.mModelInv;

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>

	pos   = vtx_input.pos;
	color = vtx_input.color.rgb;
	return pos;
}
#else
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id, float t)
{
	return pos;
}
#endif

float get_initial_modifier_factor(int ribbon_id, int segment_idx, int sub_segment_idx)
{
	float t = float(segment_idx) + float(sub_segment_idx) / float(ribbon_render_params.segment_subdivisions);
	if (segment_idx > 0)
		t += ribbon_render_params.first_segment_time_factor - 1.0;
		
	if (ribbon_render_params.export_segment_t_as_modifier_factor != 0)
		return t;
	if (ribbon_render_params.export_ribbon_t_as_modifier_factor != 0)
		return t / float(get_ribbon_segments(ribbon_id) * ribbon_render_params.segment_subdivisions);
	return 1.0;
}

float get_initial_modifier_factor_t(int ribbon_id, int segment_idx, float t)
{
	t = float(segment_idx) + t;
	if (ribbon_render_params.export_segment_t_as_modifier_factor != 0)
		return t;
	if (ribbon_render_params.export_ribbon_t_as_modifier_factor != 0)
		return t / float(get_ribbon_segments(ribbon_id) * ribbon_render_params.segment_subdivisions);
	return 1.0;
}

vec3 get_ribbon_transformed_segment_position(int ribbon_id, int segment_idx)
{
	vec3 p = ribbon_get_coords_for_segment(ribbon_id, segment_idx);

	// TODO: Apply modifiers here!

	vec3 dummy_normal = vec3(0.0);
	vec3 dummy_color = vec3(0.0);
	vec3 modified_p = apply_vtx_modifier(p, dummy_normal, dummy_color, ribbon_id, get_initial_modifier_factor(ribbon_id, segment_idx, 0));

	return vector_transform_by_mat43(modified_p, transform_params.mModelView);
}

vec3 get_ribbon_transformed_segment_position_t(int ribbon_id, int segment_idx, float sub_segment_t)
{
	vec3 p = ribbon_get_coords_for_segment_t(ribbon_id, segment_idx, sub_segment_t);

	// TODO: Apply modifiers here!

	vec3 dummy_normal = vec3(0.0);
	vec3 dummy_color = vec3(0.0);
	vec3 modified_p = apply_vtx_modifier(p, dummy_normal, dummy_color, ribbon_id, get_initial_modifier_factor_t(ribbon_id, segment_idx, sub_segment_t));

	return vector_transform_by_mat43(modified_p, transform_params.mModelView);
}

// ray-cylinder intersection as in https://iquilezles.org/articles/intersectors/
vec4 ray_cyl_intersect(in vec3 ro, in vec3 rd, in vec3 cb, in vec3 ca, float cr)
{
	vec3  oc = ro - cb;
	float caca = dot(ca, ca);
	float card = dot(ca, rd);
	float caoc = dot(ca, oc);
	float a = 1.0 - card * card;
	float b = dot( oc, rd) - caoc*card;
	float c = dot( oc, oc) - caoc*caoc - cr*cr;
	float h = b*b - a*c;
	if( h<0.0 ) return vec4(-1.0); //no intersection
	h = sqrt(h);

	float t = (-b - h) / a;
	float y = caoc + t * card;
	vec3  n = (oc + t * rd - ca * y / caca) / cr;

	return vec4(t, n.x, n.y, n.z);
}

void main() {

#ifdef DEFERRED_PASS
	instanceID = SYS_InstanceIndex;
#endif

	// fetch input data
	// when subdivision is enabled, SYS_VertexIndex extends up to the multiply of original ribbon vertices
	// when segments == n, with subdivision it is: (segments - 1) * subdivisions + 1
	
	// When doing subdivisions, last vtx will only get ribbon_sub_segment == 0, nothing else.
	// But it is important to take this case into accoint (when fetching +1 vtx) that is is not NaN or something
	// If it is anything else we can lerp between it and it will cancel out
	
	int odd_even = (SYS_VertexIndex & 1) == 0 ? 1 : -1;
	int ribbon_id;
	int ribbon_segment;
	int ribbon_sub_segment; // 0... ribbon_render_params.segment_subdivisions - 1, so the fractionally it is [0...n/(n-1)]

	if (ribbon_render_params.render_as_strips != 0)
	{
		ribbon_id = SYS_InstanceIndex;
		int n = int(SYS_VertexIndex) / 2;
		
		ribbon_segment = n / ribbon_render_params.segment_subdivisions;
		ribbon_sub_segment = n - ribbon_segment * ribbon_render_params.segment_subdivisions;
	}
	else
	{
		// in this mode we need to handle subdivisions too: ribbon_render_params.segment_subdivisions
		uint max_segments_uint = (ribbon_render_params.max_segments - 1) * ribbon_render_params.segment_subdivisions + 1;
		const int RIBBONS_PER_INSTANCE = 64;
		uint n = uint(SYS_VertexIndex) / 2;

		ribbon_id = SYS_InstanceIndex * RIBBONS_PER_INSTANCE + int(n / max_segments_uint);
		n = n % max_segments_uint;
		ribbon_segment = int(n / ribbon_render_params.segment_subdivisions);
		ribbon_sub_segment = int(n - ribbon_segment * ribbon_render_params.segment_subdivisions);
	}

	uint ribbon_segments = get_ribbon_segments(ribbon_id);
	
	// NOTE: Here it somehow prohibts shader optimization? Or just the vs <> fs ratio is so much skewed?:(
	// NOTE: With latest drivers seems to be faster again....
	if (!prt_is_alive(ribbon_id) || ribbon_segment > ribbon_segments || (ribbon_segment == ribbon_segments && ribbon_sub_segment > 0) || ribbon_segments <= 1)
	{
		gl_Position = vec4(0.0/0.0, 0.0/0.0, 0.0/0.0, 0.0/0.0);
		return;
	}
		
	if (ribbon_id == 24019)
	{
	//	gl_Position = vec4(0.0, 0.0, 0.0, -1.0);
	//	return;
	}

	//ribbon_segment = min(ribbon_segment, int(ribbon_segments));

	//vec3 vInstPosition = ribbon_get_coords_for_segment(ribbon_id, ribbon_segment);
	vec3 vInstPosition = ribbon_get_coords_for_segment_t(ribbon_id, ribbon_segment, compute_t_from_subsegment(ribbon_sub_segment));
	
	vec3 vInstNormal   = vec3(1.0, 0.0, 0.0);
	vec4 vInstColor    = prt_get_color(ribbon_id);

	vInstPosition      = apply_vtx_modifier(vInstPosition, vInstNormal, vInstColor.rgb, ribbon_id, get_initial_modifier_factor(ribbon_id, ribbon_segment, ribbon_sub_segment));

	ParticleState vState = prt_get_state(ribbon_id);

	bool is_last = ribbon_segment == ribbon_segments && ribbon_sub_segment == 0;
	bool is_last_section = ribbon_segment >= ribbon_segments - 1;
	bool is_first_section = ribbon_segment == 0;
	bool is_one_before_last = ribbon_segment == (ribbon_segments - 1) && ribbon_sub_segment == (ribbon_render_params.segment_subdivisions - 1);
	bool is_first = ribbon_segment == 0 && ribbon_sub_segment == 0;
	bool is_full = ribbon_segments == ribbon_render_params.max_segments - 1;

	vec3 extrusion = vec3(0.0);
	vec3 modelview_cylinder_axis = vec3(0.0, 0.0, 1.0);	// fallback

	int second_segment_offset = 0;				// by default go to previous segment. we will use this to shrink the 'tail' too
	int second_subsegment_offset = -1;
	bool swap_second_segment_direction = false;
	if (ribbon_sub_segment == 0 && second_subsegment_offset < 0)
	{
		second_segment_offset = -1;
		second_subsegment_offset = 0;
	}
	if (ribbon_segment == 0 && second_segment_offset < 0)
	{
		// so we can compute the axis for the cylinder. remember to reverse it
		swap_second_segment_direction = true;
		second_subsegment_offset = 1;
		if (second_subsegment_offset >= ribbon_render_params.segment_subdivisions)
		{
			second_subsegment_offset = 0;
			second_segment_offset = 1;
		}
	}

	if (true)
	{
		//float shrink_factor = ribbon_render_params.first_segment_time_factor;
		float ribbon_t;
		if (is_first_section)
		{
			// compensate for length of first section
			ribbon_t = float(ribbon_sub_segment) * ribbon_render_params.first_segment_time_factor / float(ribbon_render_params.segment_subdivisions); 
		}
		else
		{
			ribbon_t = float(ribbon_segment) + float(ribbon_sub_segment) / float(ribbon_render_params.segment_subdivisions);
			ribbon_t -= 1.0 - ribbon_render_params.first_segment_time_factor; // as the first section grows, we reduce the compensation
		}
		
		float curve_mirror_point = (float(ribbon_segments) - 0.5 + ribbon_render_params.first_segment_time_factor) * 0.45;
		if (ribbon_segments > 3 && ribbon_t > curve_mirror_point)
		{
			ribbon_t = curve_mirror_point - (ribbon_t - curve_mirror_point) * 0.75; // mirror. this is simplistic as our tail will also be shrinking, but good enough?
		}
		
		float shrink_factor = 1.0 - exp(-ribbon_t * 2.0);
		float min_shrink_modifier = 0.05;

		// camera-space positions. when computing extrusions just use XY plane
		//vec3 currPosition = get_ribbon_transformed_segment_position(ribbon_id, ribbon_segment);
		//vec3 nextPosition = get_ribbon_transformed_segment_position(ribbon_id, ribbon_segment + second_segment_offset);
		float t_shift = (1.0 / float(ribbon_render_params.segment_subdivisions)) * 0.05;
		vec3 currPosition = get_ribbon_transformed_segment_position_t(ribbon_id, ribbon_segment, compute_t_from_subsegment(ribbon_sub_segment) - t_shift);
		vec3 nextPosition = get_ribbon_transformed_segment_position_t(ribbon_id, ribbon_segment + second_segment_offset, compute_t_from_subsegment(ribbon_sub_segment + second_subsegment_offset) + t_shift);
		
		// sometimes segments are too short/zero-length. try to handle this but this is kind of lost cause:(
		modelview_cylinder_axis = currPosition - nextPosition;

		[[branch]]
		if (length(modelview_cylinder_axis) <= 0.0001)
		{
			if (ribbon_segment > 1)
			{
				// we can extrapolate, it is fine
				nextPosition = get_ribbon_transformed_segment_position_t(ribbon_id, ribbon_segment + second_segment_offset, ribbon_sub_segment + second_subsegment_offset * 2);
				modelview_cylinder_axis = currPosition - nextPosition;
			}
		}

		modelview_cylinder_axis = normalize(modelview_cylinder_axis);
		vec2 dir2d = normalize(currPosition.xy - nextPosition.xy);	// NOTE: Don't just normalize in 3D and take XY
		
		if (swap_second_segment_direction)
		{
			modelview_cylinder_axis *= -1.0;
			dir2d *= -1.0;
		}
		
		dir2d = vec2(-dir2d.y, dir2d.x);
		vec3 dir = vec3(dir2d, 0.0);

		// if we are last segment then interpolate to shrink it towards previous. smoother end of trail motion
		if (ribbon_segment == ribbon_segments && ribbon_segments > 1) {
			//vInstPosition = mix(vInstPosition, nextPosition, ribbon_render_params.first_segment_time_factor);
			//dir *= min_shrink_modifier;
		}

		// move tail towards the previous vtx for smoother motion when ribbon is full (do it after all the extruding computations)
		// tail constitutes of all the subdivisions in the last section
		[[branch]]
		if (is_full && is_last_section)
		{
			float t = ribbon_render_params.first_segment_time_factor;
			vec3 prevInstPosition = ribbon_get_coords_for_segment_t(ribbon_id, int(ribbon_segments - 1), 0);
			prevInstPosition = apply_vtx_modifier(prevInstPosition, vInstNormal, vInstColor.rgb, ribbon_id, get_initial_modifier_factor(ribbon_id, ribbon_segment, ribbon_sub_segment));
			vInstPosition = mix(vInstPosition, prevInstPosition, t);
		}
		
		dir *= max(min_shrink_modifier, shrink_factor);

		//if (ribbon_segment < ribbon_segments) {
			extrusion = dir * ribbon_render_params.thickness;
		//}

		//vInstColor.rgb = vec3((currPosition.xy - nextPosition.xy) * 0.1 * 0.5 + 0.5, 0.0);
		//vInstColor.rgb = vec3(length(currPosition.xy - nextPosition.xy));

	}
	
	// revert the xylinder axis for the first segment
	if (ribbon_segment == 0)
		modelview_cylinder_axis *= -1.0;

	if (!prt_is_alive(ribbon_id))
	{
		gl_Position = vec4(0.0/0.0, 0.0/0.0, 0.0/0.0, -1.0);
		return;
	}

	//
	// NOTE: Export main line coordinates as world coords. In addition export interpolant for the left-to-right side of the strip
	// so we can try to reconstruct proper 'tube' in ps.
	//

	vec3 pos = vInstPosition;
#ifdef DEFERRED_PASS	
	vtx_output.vLocalPos = pos;
#endif	
	vec3 vPos1 = pos;
	vec3 vPos = vector_transform_by_mat43(vPos1, transform_params.mModelView);

	// check extrusion's projection. should we limit this to some range? TODO
	if (vPos.z > 1.0)
	{
		// NOTE: No idea why jitter breaks the extrusion check for now
		mat4 projection_no_jitter = transform_params.mProjection;
		projection_no_jitter[2][0] = 0.0;
		projection_no_jitter[2][1] = 0.0;

		vec4 projected_extrusion = vector_transform_by_mat_projection(vec3(extrusion.xy, vPos.z), projection_no_jitter);
		
		projected_extrusion.xyz = projected_extrusion.xyz / projected_extrusion.w;
		float min_extrusion_size = 0.00065;		// make this resolution dependant, but should be fine for anything > 1080.0
		float projected_extrusion_size = length(projected_extrusion.xy);
		if (projected_extrusion_size > 0.0)		// should always be the case...
		{
			if (projected_extrusion_size < min_extrusion_size)
			{
				float scale = min_extrusion_size / projected_extrusion_size;
				extrusion *= scale;
				//vInstColor.rgb = vec3(1.0, 0.0, 0.0); // debug
			}
		}
	}

	// adjust extrusion by remaining life (bit smoother disapearing)
	float remaining_lifetime_based_extrusion_modifier = 1.0;
	float remaining_lifetime_margin = 0.2;
	if (vState.life_time > vState.life_span - remaining_lifetime_margin)
	{
		remaining_lifetime_based_extrusion_modifier = vState.life_time - (vState.life_span - remaining_lifetime_margin);
		remaining_lifetime_based_extrusion_modifier /= (vState.life_span - remaining_lifetime_margin);
		remaining_lifetime_based_extrusion_modifier = 1.0 - remaining_lifetime_based_extrusion_modifier;
	}
	else if (vState.life_time < remaining_lifetime_margin)
	{
		remaining_lifetime_based_extrusion_modifier = vState.life_time;
		remaining_lifetime_based_extrusion_modifier /= remaining_lifetime_margin;
		remaining_lifetime_based_extrusion_modifier = 1.0 - remaining_lifetime_based_extrusion_modifier;
	}
	
	remaining_lifetime_based_extrusion_modifier = clamp(remaining_lifetime_based_extrusion_modifier, 0.0, 1.0);
	vPos += extrusion * remaining_lifetime_based_extrusion_modifier * float(odd_even);

#ifdef DEFERRED_PASS
	MaterialPropertiesGPU material = materials.material_properties[ribbon_render_params.material_index];

	// NOTE: We are passing slightly incorrect world pos because of how this displacement is applied:()
	vtx_output.vWorldPos      = vector_transform_by_mat43(vPos1 + extrusion * float(odd_even), transform_params.mModel);
	vtx_output.vWorldPos      = vtx_output.vWorldPos - transform_params.vCameraPosition;
	vtx_output.vWorldNorm.xyz = vec3(vInstPosition);
	vtx_output.vNorm          = vInstNormal;
	vtx_output.vCoords        = vPos.xyz;

	// In case of transparent material do the shading in VS (TODO: move to PS?)
	// Otherwise we fake the cylindrical shape by outputting position across the ribbon

#ifdef RIBBON_MATERIAL_TRANSPARENT
	vInstColor = calculate_lighting(vInstColor, material);
#else
	vtx_output.vNorm         = vec3(odd_even, 0.0, -0.5);	// slightly towards camera

	// build the cylinder in the model space. we are trying to move the extruded (in camera space) shape back into proper world space
	vec3 model_cylinder_axis = vector_transform_by_mat33(modelview_cylinder_axis, transform_params.mViewInv);
	vec3 model_on_cylinder_pos = vector_transform_by_mat43(vPos, transform_params.mViewInv);
	vec3 model_cylinder_pos  = vector_transform_by_mat43(vInstPosition, transform_params.mModel);
	vec4 cylinder_intersect  = ray_cyl_intersect(
		transform_params.vCameraPosition,
		normalize(model_on_cylinder_pos - transform_params.vCameraPosition),
		model_cylinder_pos,
		model_cylinder_axis,
		length(extrusion) * 2.0);	// NOTE: large radius for the cylinder doesn't really hurt us that much but is safer

	vtx_output.vWorldNorm       = -vec3(cylinder_intersect.yzw);
	if (cylinder_intersect.x == -1.0)
	{
		//vInstColor.rgb = vec3(1.0, 0.0, 0.0);	// debug
	}
#endif

	float life_time_factor = min(1.0, vState.life_time / vState.life_span);
	//if (life_time_factor < 0.1)
	life_time_factor = 1.0 - sin(life_time_factor * M_PI * 0.5);
	vInstColor.rgb *= 1.0 - pow(life_time_factor, 4.0);
	vtx_output.vColor = f16vec4(vInstColor);

#endif // DEFERRED_PASS

	gl_Position = vector_transform_by_mat_projection(vPos, transform_params.mProjection);
}

