
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "clax.h"
#include "render.h"

char*   outBuffer;
char*   nzBuffer;
char*   trBuffer;
int     bufferWidth, bufferHeight;

#define  MAXFACES   10000

c_DSORT*  sortList;
c_DSORT*  tmpList;
c_DFACE*  faceList;
int       nFaces;


#define ByteOf(x) (((x) >> bitsOffset) & 0xff)

static
void
radix( short bitsOffset, ulong N, c_DSORT *source, c_DSORT *dest )
{
  // suppressed the need for index as it is reported in count
  ulong    count[256];

  // added temp variables to simplify writing, understanding and compiler optimization job
  // most of them will be allocated as registers
  ulong    *cp;
  c_DSORT  *sp;
  ulong    s, c, i;

  // faster than MemSet
  cp = count;
  for (i = 256; i > 0; --i, ++cp)
    *cp = 0;

  // count occurences of every byte value
  sp = source;
  for (i = N; i > 0; --i, ++sp) {
    cp = count + ByteOf( sp->z );
    (*cp)++;
  }

  // transform count into index by summing elements and storing into same array
  s = 0;
  cp = count;
  for (i = 256; i > 0; --i, ++cp) {
    c = *cp;
    *cp = s;
    s += c;
  }

  // fill dest with the right values in the right place
  sp = source;
  for (i = N; i > 0; --i, ++sp) {
    cp = count + ByteOf( sp->z );
    dest[*cp].z    = sp->z;
    dest[*cp].face = sp->face;
    (*cp)++;
  }
}

void
sortPolys()
{
  radix( 0, nFaces, sortList, tmpList );
  radix( 8, nFaces, tmpList, sortList );
  radix( 16, nFaces, sortList, tmpList );
  radix( 24, nFaces, tmpList, sortList );
}



void
drawTriangle( c_VERTEX* v0, c_VERTEX* v1, c_VERTEX* v2, int color )
{

    double denom = (v0->sx - v2->sx) * (v1->sy - v2->sy) -
                   (v1->sx - v2->sx) * (v0->sy - v2->sy);
    if( denom == 0.0 )
      return;

    long didx, dnzdx;
    didx = (long)(((v0->i - v2->i) * (v1->sy - v2->sy) - (v1->i - v2->i) * (v0->sy - v2->sy)) / denom * 65536.0);
    dnzdx = (long)(((v0->nz - v2->nz) * (v1->sy - v2->sy) - (v1->nz - v2->nz) * (v0->sy - v2->sy)) / denom * 65536.0);

    c_DVERTEX*  v;


    // gouraud
    v = &faceList[nFaces].v[0];
	v->x = (long)(v0->sx * 65536.0f);
	v->y = (long)(v0->sy * 65536.0f);
	v->i = (long)(v0->i * 65536.0f);
	v->nz = (long)(v0->nz * 65536.0f);
    v->i += (color << 16);        // add base color of gradient

    v = &faceList[nFaces].v[1];
	v->x = (long)(v1->sx * 65536.0f);
	v->y = (long)(v1->sy * 65536.0f);
	v->i = (long)(v1->i * 65536.0f);
	v->nz = (long)(v1->nz * 65536.0f);
    v->i += (color << 16);

    v = &faceList[nFaces].v[2];
	v->x = (long)(v2->sx * 65536.0f);
	v->y = (long)(v2->sy * 65536.0f);
	v->i = (long)(v2->i * 65536.0f);
	v->nz = (long)(v2->nz * 65536.0f);
    v->i += (color << 16);

    faceList[nFaces].didx = didx;
    faceList[nFaces].dnzdx = dnzdx;

    // farthest point
    long  z = *((long *)&v0->sz);
    if (*((long *)&v1->sz) < z )
      z = *((long *)&v1->sz);
    if (*((long *)&v2->sz) < z )
      z = *((long *)&v2->sz);
    sortList[nFaces].z = z;
    sortList[nFaces].face = &faceList[nFaces];

    nFaces++;
    // add new poly
}


inline long ceil(long x)
{
    x +=  0xffff;
    return (x >> 16);
}

void
flushPolys()
{
  int        i;
  c_DFACE*   face;

  sortPolys();

  for ( i = 0; i < nFaces; i++ ) {
    face = sortList[i].face;
    DrawGouraudPoly( face->v, 3, face->didx, face->dnzdx );
  }

  nFaces = 0;  
}





//
//   VIEWPORT
//

int
viewportOpen( int width, int height, int maxFaces, c_VIEWPORT* vp )
{

  vp->iWidth = width;
  vp->iHeight = height;

  if ( maxFaces == -1 )
    maxFaces = MAXFACES;

  vp->outBuffer = new char [vp->iWidth * vp->iHeight];
  if ( !vp->outBuffer ) {
    return clax_err_nomem;
  }

  vp->nzBuffer = new char [vp->iWidth * vp->iHeight];
  if ( !vp->nzBuffer ) {
    delete [] vp->outBuffer;
    return clax_err_nomem;
  }

  vp->trBuffer = new char [vp->iWidth * vp->iHeight];
  if ( !vp->trBuffer ) {
    delete [] vp->outBuffer;
    delete [] vp->nzBuffer;
    return clax_err_nomem;
  }

  vp->sortList = new c_DSORT [maxFaces];
  if ( !vp->sortList ) {
    delete [] vp->outBuffer;
    delete [] vp->nzBuffer;
    delete [] vp->trBuffer;
    return clax_err_nomem;
  }

  vp->tmpList = new c_DSORT [maxFaces];
  if ( !vp->tmpList ) {
    delete [] vp->outBuffer;
    delete [] vp->nzBuffer;
    delete [] vp->trBuffer;
    delete [] vp->sortList;
    return clax_err_nomem;
  }

  vp->faceList = new c_DFACE [maxFaces];
  if ( !vp->tmpList ) {
    delete [] vp->outBuffer;
    delete [] vp->nzBuffer;
    delete [] vp->trBuffer;
    delete [] vp->sortList;
    delete [] vp->tmpList;
    return clax_err_nomem;
  }

  return clax_err_ok;
}


void
viewportClose( c_VIEWPORT* vp )
{
  if ( vp->outBuffer )
    delete [] vp->outBuffer;
  if ( vp->nzBuffer )
    delete [] vp->nzBuffer;
  if ( vp->trBuffer )
    delete [] vp->trBuffer;
  if ( vp->sortList )
    delete [] vp->sortList;
  if ( vp->tmpList )
    delete [] vp->tmpList;
  if ( vp->faceList )
    delete [] vp->faceList;
}

void
viewportSetactive( c_VIEWPORT* vp )
{  
  outBuffer = vp->outBuffer;
  nzBuffer = vp->nzBuffer;
  trBuffer = vp->trBuffer;

  bufferWidth = vp->iWidth;
  bufferHeight = vp->iHeight;

  sortList = vp->sortList;
  tmpList = vp->tmpList;
  faceList = vp->faceList;
  
  nFaces = 0;
}

void
viewportGetactive( c_VIEWPORT* vp )
{  
  vp->outBuffer = outBuffer;
  vp->nzBuffer = nzBuffer;
  vp->trBuffer = trBuffer;

  vp->iWidth = bufferWidth;
  vp->iHeight = bufferHeight;

  vp->sortList = sortList;
  vp->tmpList = tmpList;
  vp->faceList = faceList;
}
