/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/core/cRegistry.h>

// extern includes
#include <caosGL/gfx/cGroup.h>
#include <caosGL/gfx/cSceneGraph.h>
#include <caosGL/gfx/cAnimCurveReader.h>
#include <caosGL/gfx/cAnimCurve.h>
#include <caosGL/core/tinyxml.h>
#include <caosGL/core/cUtil.h>
#include <math.h>

#include <caosGL/core/cScript.h>
#include <caosGL/core/cParser.h>
#include <caosGL/core/cEvent.h>

using namespace caosGL::gfx;

namespace caosGL {
	namespace core {
		/**
		 *<br> class:		cScript
		 *<br> namespace:	caosGL::core
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		
		 *
		 */

	    static log4cpp::Category& cat = log4cpp::Category::getInstance ("sys.cScript");
		cScript * cScript::_instance = cNULL;
		/********************************************************************************************/
		cScript::cScript () {} 
		/********************************************************************************************/
		cScript::~cScript () {} 

		/********************************************************************************************/
		cScript * cScript::instance () {
			if (_instance == cNULL) {
				_instance = makeInstance ();
			}
			return _instance;
		}

		/********************************************************************************************/
		cScript * cScript::makeInstance () {
			return new cScript ();
		}

		/********************************************************************************************/
		tBool cScript::readScript (const string & name) {
			TiXmlDocument * doc = new TiXmlDocument (name);
			if (!doc->LoadFile()) {
				cat.emerg ("Could not load the scriptFile: %s", name.c_str ());
				delete doc;
				return false;
			}

			readScript (doc);

			delete doc;

			return true;
		}

		/********************************************************************************************/
		tBool cScript::readScript (TiXmlDocument * doc) {
			TiXmlElement * caosGL = doc->FirstChildElement ("demo");

			for (TiXmlElement * includeDoc = caosGL->FirstChildElement ("include"); 
					includeDoc; includeDoc = includeDoc->NextSiblingElement ("include")) {
				string file;
				for (TiXmlAttribute * attr = includeDoc->FirstAttribute (); attr ; attr = attr->Next ()) {
					string attrName = attr->Name ();
					if (attrName == "file")
						file = attr->Value ();
				}
				if (!readScript (file))
					return false;
			}

			// events go first of all!			
			readEvents (caosGL->FirstChildElement ("events"));

			TiXmlElement * sceneGraph = caosGL->FirstChildElement ("sceneGraph");
			if (sceneGraph) {
				for (TiXmlElement * node = sceneGraph->FirstChildElement ("node"); 
						node; node = node->NextSiblingElement ("node")) {
					readNode (node, cSceneGraph::getRootNode ());
				}
			}
			readCurves (caosGL->FirstChildElement ("curves"));

			return true;
		}

		// business end of stuff...
		/********************************************************************************************/
		tBool cScript::readNode (TiXmlElement * node, cGroup * gr) {
			if (!node) return false;

			string name;
			string type;
			tFloat start = -HUGE_VAL;
			tFloat end = HUGE_VAL;

			// get all the attributes, but for now, i'm only going to use 'name' and 'type'
			for (TiXmlAttribute * attr = node->FirstAttribute (); attr ; attr = attr->Next ()) {
				string attrName = attr->Name ();
				if (attrName == "name")
					name = attr->Value ();
				else if (attrName == "type")
					type = attr->Value ();
				else if (attrName == "start")
					start = cParser::parseDouble (attr->Value ());
				else if (attrName == "end")
					end = cParser::parseDouble (attr->Value ());
			}

			// create and add the node to the group or to the scene graph if the group is null
			cBaseNode * bn = cRegistry::createNode (type, name, gr);

			// check to see if it was created or not
			if (!bn) {
				cat.error ("Error in node creation, won't descend further down the tree.");
				return false;
			}

			bn->set ('strt', start);
			bn->set ('end', end);

			gr->addNode (bn);

			// read it's params first, befores descending down the tree structure
			for (TiXmlElement * param = node->FirstChildElement ("param");
					param; param = param->NextSiblingElement ("param")) {
				tDWord pName;
				string pValue;
				for (TiXmlAttribute * pAttr = param->FirstAttribute (); pAttr ; pAttr = pAttr->Next ()) {
					string attrName = pAttr->Name ();
					if (attrName == "name")
						pName = cUtil::doFli (pAttr->Value ());
					else if (attrName == "value")
						pValue = pAttr->Value ();
				}
				bn->set (pName, pValue);
			}
			

			// to add subNodes to this node, we need it to be a cGroup.
			cGroup * gr_ = dynamic_cast <cGroup *> (bn);						
			for (TiXmlElement * subNode = node->FirstChildElement ("node");
					subNode; subNode = subNode->NextSiblingElement ("node")) {
				// if the newly created node is not a cGroup, 'gr' will be null
				// and we should inform this situation
				if (!gr_) {
					cat.error ("Can't add subNode to non-cGroup node %s", bn->toString ().c_str ());
					break;
				}
				readNode (subNode, gr_);
			}

			return true;
		}

		/********************************************************************************************/
		tBool cScript::readCurves (TiXmlElement * node) {
			if (!node) return false;
			cRegistry::addEvaluatables (cAnimCurveReader::readCurves (node));
			return true;
		}

		/********************************************************************************************/
		tBool cScript::readEvents (TiXmlElement * node) {
			if (!node) return false;
			
			cEvent * e = new cEvent ("eventDriverXXX", cSceneGraph::getRootNode ());

			for (TiXmlElement * event = node->FirstChildElement ("event");
					event; event = event->NextSiblingElement ("event")) {
				string w,v;
				tDWord n;
				tFloat t;
				for (TiXmlAttribute * attr = event->FirstAttribute (); attr ; attr = attr->Next ()) {
					string attrName = attr->Name ();
					if (attrName == "who")
						w = attr->Value ();
					else if (attrName == "name")
						n = cUtil::doFli (attr->Value ());
					else if (attrName == "value")
						v = attr->Value ();
					else if (attrName == "time")
						t = cParser::parseFloat (attr->Value ());
				}
				e->addEvent (w,n,v,t);
			}

			cSceneGraph::getRootNode ()->addNode (e);

			return true;
		}
	}
}

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof