// ------------------------------------------------------
// Synth.cpp
// CopyRight (C) 2004 Franck "hitchhikr" Charlet / TRSI.

// ------------------------------------------------------
// Includes
#define DIRECTSOUND_VERSION 0x0800
#include <DSound.h>
#include "Synth.h"
#include <stdio.h>
#include <math.h>

// ------------------------------------------------------
// Constants

// 78 -> 48kHz; 90 -> 100kHz
#define MAXNOTE	120	

// ------------------------------------------------------
// Variables
int Freqs_Table[] = {
	0x00000417,0x0000455,0x00000497,0x000004DD,0x00000527,0x00000575,0x000005C8,0x00000620,0x0000067D,0x000006E0,0x00000749,0x000007B8,
	0x0000082D,0x00008A9,0x0000092D,0x000009B9,0x00000A4D,0x00000AEA,0x00000B90,0x00000C40,0x00000CFA,0x00000DC0,0x00000E91,0x00000F6F,
	0x0000105A,0x0001153,0x0000125B,0x00001372,0x0000149A,0x000015D4,0x00001720,0x00001880,0x000019F5,0x00001B80,0x00001D23,0x00001EDE,
	0x000020B4,0x00022A6,0x000024B5,0x000026E4,0x00002934,0x00002BA7,0x00002E40,0x00003100,0x000033EA,0x00003700,0x00003A45,0x00003DBC,
	0x00004168,0x000454C,0x0000496B,0x00004DC8,0x00005268,0x0000574F,0x00005C80,0x00006200,0x000067D3,0x00006E00,0x0000748A,0x00007B79,
	0x000082D0,0x0008A97,0x000092D5,0x00009B90,0x0000A4D0,0x0000AE9D,0x0000B8FF,0x0000C3FF,0x0000CFA7,0x0000DC00,0x0000E915,0x0000F6F1,
	0x000105A0,0x001152F,0x000125AA,0x00013721,0x000149A1,0x00015D3A,0x000171FF,0x000187FF,0x00019F4E,0x0001B800,0x0001D22A,0x0001EDE2,
	0x00020B40,0x0022A5D,0x00024B54,0x00026E41,0x00029341,0x0002BA75,0x0002E3FD,0x00030FFE,0x00033E9C,0x00037000,0x0003A454,0x0003DBC4,
	0x00041681,0x00454BB,0x000496A9,0x0004DC82,0x00052683,0x000574EA,0x0005C7FA,0x00061FFB,0x00067D38,0x0006E000,0x000748A8,0x0007B789,
	0x00082D01,0x008A976,0x00092D52,0x0009B904,0x000A4D05,0x000AE9D4,0x000B8FF5,0x000C3FF6,0x000CFA70,0x000DC000,0x000E914F,0x000F6F11
};
int Mixing_buffer_Stereo[(44100 * 2) + 4];
LPDIRECTSOUND DSound_Context;
DSBUFFERDESC Sound_Buffer_Desc;
LPDIRECTSOUNDBUFFER Sound_Buffer;
WAVEFORMATEX Wave_Format;
DWORD Buffer_Pos;
DWORD Old_Buffer_Pos;
LPVOID Audio_Ptr1;
DWORD Audio_Bytes1;
LPVOID Audio_Ptr2;
DWORD Audio_Bytes2;
int Render_Wave;
int Song_Position;
int Song_Restart;
int Cur_Ticks;
int Fx_Ticks;
int Speed_Tick;
int Arpeggio_Counter;
int Tempo = 32;
int Fx_Tempo = 125;
int Tempo_Ticks;
int Fx_Tempo_Ticks;
int Synth_Play_State;
int BufferSize;
int Max_Buffer = 44100;
void (__stdcall *Synth_CallBack)(int Datas);
float Flt32767 = 32767.0f;
float Flt32768 = 32768.0f;
float Fltm1 = -1.0f;
float Flt05 = 0.5f;
float Fltm05 = -0.5f;
float Flt005 = 0.05f;
float Flt1 = 1.0f;
float Flt025 = 0.25f;
float Fltm005 = -0.05f;
DWORD Random_Seed = 0x12341234;
float FltPI2 = 6.2831853071f;
float FltPI = 3.1415926535f;

float Flt0000023 = 0.000023f;
float Fltm2 = -2.0f;
float Flt116 = 1.16f;
float Flt015 = 0.15f;
float Flt0350130 = 0.35f;
float Flt03 = 0.3f;
float Flt100 = 100.0f;
float Flt001 = 0.01f;
float Flt00001 = 0.0001f;
float Flt0000001 = 0.0000001f;
float Flt01 = 0.1f;
float Flt016 = 0.0f;

int Size_Sequence_Dat;
int Instr_Dat_Number;
int Instr_Dat_Vars;
int Size_Instrument_Vars;

WORD *Ptr_Spectrum_Buffer;
WORD Spectrum_Buffer[44100];

int Play_Type;
LPINSTRUMENT Current_Instrument;

int OnOff_Status[8];

// Default panning for instruments
float Instrument_Pan = 0.5f;		// 0.0 = left 1.0 = right
int Cur_Seq_Play_Pos;

CList Instruments;					// INSTRUMENT
CList Instruments_Dats;				// INSTRUMENT_DAT
CList *Positions[8];
CList Sequences;

SEQUENCE Sequences_Dats[8];
SEQUENCE Dummy_Sequence_Dats;

int Ptr_Sequence;
float Global_Volume;

// Instruments
float Final_Signal;
int Current_Instr_Note = 60 + 1;
int Instr_Datas_Copied;

double coeff[5][11]= {
	// A
	{	3.11044e-06,
		8.943665402, -36.83889529, 92.01697887, -154.337906, 181.6233289,
		-151.8651235, 89.09614114, -35.10298511, 8.388101016, -0.923313471
	},
	// E
	{	4.36215e-06,
		8.90438318, -36.55179099, 91.05750846, -152.422234, 179.1170248,
		-149.6496211,87.78352223, -34.60687431, 8.282228154, -0.914150747
	},
	// I
	{	3.33819e-06,
		8.893102966, -36.49532826, 90.96543286, -152.4545478, 179.4835618,
		-150.315433, 88.43409371, -34.98612086, 8.407803364, -0.932568035
	},
	// O
	{	1.13572e-06,
		8.994734087, -37.2084849, 93.22900521, -156.6929844, 184.596544,
		-154.3755513, 90.49663749, -35.58964535, 8.478996281, -0.929252233
	},
	// U
	{	4.09431e-07,
		8.997322763, -37.20218544, 93.11385476, -156.2530937, 183.7080141,
		-153.2631681, 89.59539726, -35.12454591, 8.338655623, -0.910251753
	}
};

int Reverb_Delays[6] = { 0, 13, 141, 310, 71, 465 };
//int Reverb_Delays[6] = { 0, 0, 0, 0, 0, 0};

#define GKA_SAMPLE 2 // 1=8bit, 2=16bit
#define GKA_CHANNELS 1 // 1=mono, 2=stereo
#define GKA_FREQ 44100 // frequency (samples/second)
#define GKA_SIZE (4410 * 16 * 76)

struct {
	char Riff[4];
	DWORD Size;
	char Wave[4];
	char Fmt[4];
	DWORD Num;
	WORD Format;
	WORD Channels;
	DWORD Freq;
	DWORD ABPS;
	WORD BlockAlign;
	WORD BitDepth;
	char Data[4];
	DWORD DataSize;
} gkaWavhead = {
	{'R', 'I', 'F', 'F'},
	36 + (2 * 2 * GKA_SIZE),
	{'W', 'A', 'V', 'E'},
	{'f', 'm', 't', ' '},
	0x10,
	1,
	2,
	44100,
	2 * 2 * 44100,
	2 * 2,
	2 * 8,
	{'d', 'a', 't', 'a'},
	2 * 2 * GKA_SIZE,
};

// ------------------------------------------------------
// Functions
void __stdcall Process_Sound_Dat(LPVOID Bytes, DWORD DatSize);
int __stdcall Get_Positions_Number(void);
int __stdcall Get_Sequence_Dat(int Seq_Number);
int __stdcall Get_Position_Dat(int Pos_Number, int Pos_Index);
int __stdcall Get_Instrument_Dat(LPINSTRUMENT Cur_Instrument, int Instr_Number);
int __stdcall Get_Instrument_Comb(LPINSTRUMENT Cur_Instrument);
int __stdcall Get_Instrument(int Instr_Number);
void __stdcall Wave_WhiteNoise(void);
void __stdcall Set_Delay(LPSEQUENCE Seq_Dats, float Delay_Distance, float Delay_Feedback, float Delay_Pan);
void __stdcall Set_Reverb(LPSEQUENCE Seq_Dats, float Reverb_Distance, float Reverb_Feedback, float Reverb_Pan);
void __stdcall Clamp_Value(void);
void Clear_Sequence_Dats(LPSEQUENCE Seq_To_Clear);

// ------------------------------------------------------
// Name: Synth_Init()
// Desc: Init the synthetizer
int __stdcall Synth_Init(HWND hWnd, void (__stdcall *CallBack)(int Datas)) {
	int Init_Return = FALSE;
	int i;

	if(DirectSoundCreate(NULL, &DSound_Context, NULL) == DS_OK) {
		if(DSound_Context->SetCooperativeLevel(hWnd, DSSCL_EXCLUSIVE) == DS_OK) {
			Sound_Buffer_Desc.dwSize = sizeof(Sound_Buffer_Desc);
			Sound_Buffer_Desc.dwFlags = DSBCAPS_STICKYFOCUS;
			// 44100 hz
			Sound_Buffer_Desc.dwBufferBytes = Max_Buffer;
			Wave_Format.wFormatTag = WAVE_FORMAT_PCM;
			// Mono / Stereo
			Wave_Format.nChannels = 2;
			Wave_Format.nSamplesPerSec = 44100;
			Wave_Format.nAvgBytesPerSec = (44100 * ((16 * 2) >> 3));
			Wave_Format.nBlockAlign = ((16 * 2) >> 3);
			// 8 / 16 bits
			Wave_Format.wBitsPerSample = 16;
			Sound_Buffer_Desc.lpwfxFormat = &Wave_Format;
			if(DSound_Context->CreateSoundBuffer(&Sound_Buffer_Desc, &Sound_Buffer, NULL) == DS_OK) {
				if(Sound_Buffer->Play(0, 0, DSBPLAY_LOOPING) == DS_OK) {
					float Freq_Value;
					int Int_Freq_Value;
					for(i = 0; i < MAXNOTE; i++ ) {
						Freq_Value = 440.0f * powf(2, ((float) i + 15) / 12);
						_asm {
							fld		dword ptr [Freq_Value]
							fistp	dword ptr [Int_Freq_Value]
						}
						Freqs_Table[i] = Int_Freq_Value;
					}
					Synth_CallBack = CallBack;
					Init_Return = TRUE;
				}
			}
		}
	}
	// 16 buffers to allocate (~3 megs)
	Dummy_Sequence_Dats.Seq_Delay_Left_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
	Dummy_Sequence_Dats.Seq_Delay_Right_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
	Dummy_Sequence_Dats.Seq_Reverb_Left_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
	Dummy_Sequence_Dats.Seq_Reverb_Right_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);

	for(i = 0; i < 8; i++) {
		Sequences_Dats[i].Seq_Delay_Left_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
		Sequences_Dats[i].Seq_Delay_Right_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
		Sequences_Dats[i].Seq_Reverb_Left_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
		Sequences_Dats[i].Seq_Reverb_Right_Buffer = (float *) GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, 44100 * 4);
	}

	Tempo_Ticks = Get_Mix_Buffer_Len(Tempo);
	// Arpeggio is fixed at 125 BPM so it doesn't sound like shit at hight/lower tempi
	Fx_Tempo_Ticks = Get_Mix_Buffer_Len(Fx_Tempo);
	Synth_Play_State = TRUE;
	return(Init_Return);
}

// ------------------------------------------------------
// Name: Synth_Stop()
// Desc: Stop the synthetizer
void __stdcall Synth_Stop(void) {
	Synth_Play_State = FALSE;
	if(Sound_Buffer) {
		Sound_Buffer->Stop();
		Sound_Buffer->Release();
	}
	if(DSound_Context) DSound_Context->Release();
	if(Dummy_Sequence_Dats.Seq_Delay_Left_Buffer) GlobalFree(Dummy_Sequence_Dats.Seq_Delay_Left_Buffer);
	if(Dummy_Sequence_Dats.Seq_Delay_Right_Buffer) GlobalFree(Dummy_Sequence_Dats.Seq_Delay_Right_Buffer);
	if(Dummy_Sequence_Dats.Seq_Reverb_Left_Buffer) GlobalFree(Dummy_Sequence_Dats.Seq_Reverb_Left_Buffer);
	if(Dummy_Sequence_Dats.Seq_Reverb_Right_Buffer) GlobalFree(Dummy_Sequence_Dats.Seq_Reverb_Right_Buffer);

	for(int i = 0; i < 8; i++) {
		if(Sequences_Dats[i].Seq_Delay_Left_Buffer) GlobalFree(Sequences_Dats[i].Seq_Delay_Left_Buffer);
		if(Sequences_Dats[i].Seq_Delay_Right_Buffer) GlobalFree(Sequences_Dats[i].Seq_Delay_Right_Buffer);
		if(Sequences_Dats[i].Seq_Reverb_Left_Buffer) GlobalFree(Sequences_Dats[i].Seq_Reverb_Left_Buffer);
		if(Sequences_Dats[i].Seq_Reverb_Right_Buffer) GlobalFree(Sequences_Dats[i].Seq_Reverb_Right_Buffer);
	}
}

// ------------------------------------------------------
// Name: Synth_Play()
// Desc: Replay routine
void __stdcall Synth_Play(void) {
	DWORD Bytes_To_Lock;

	if(Synth_Play_State) {
		Sound_Buffer->GetCurrentPosition(&Buffer_Pos, NULL);
		// Update sound buffer position
		_asm {
			mov		eax,[Buffer_Pos]
			sub		eax,[Old_Buffer_Pos]
			jns		Reset_Buffer_Pos
			add		eax,[Max_Buffer]
Reset_Buffer_Pos:
			mov		[Bytes_To_Lock],eax
		}
		Sound_Buffer->Lock(Old_Buffer_Pos, Bytes_To_Lock, &Audio_Ptr1, &Audio_Bytes1, &Audio_Ptr2, &Audio_Bytes2, 0);
		_asm {
			mov		eax,[Buffer_Pos]
			mov		[Old_Buffer_Pos],eax
		}

		Ptr_Spectrum_Buffer = (WORD *) &Spectrum_Buffer;
		memset(&Spectrum_Buffer, 0, sizeof(Spectrum_Buffer));

		// Save old position
		_asm {
			push	dword ptr [Audio_Bytes1]
			push	dword ptr [Audio_Ptr1]
			call	Process_Sound_Dat
			push	dword ptr [Audio_Bytes2]
			push	dword ptr [Audio_Ptr2]
			call	Process_Sound_Dat
		}
		// unlock buffer now
		Sound_Buffer->Unlock(Audio_Ptr1, Audio_Bytes1, Audio_Ptr2, Audio_Bytes2);
	}
}

// ------------------------------------
// Process song datas
void __stdcall Process_Sound_Dat(LPVOID Bytes, DWORD DatSize) {
	int i;

	switch(Play_Type) {
		case No_Play:
			_asm {
				mov		edi,[Bytes]
				mov		ecx,[DatSize]
				shr		ecx,1
				xor		ax,ax
				rep		stosw
			}
			break;

		case Play_Instrument:
		case Play_Sequence:
		case Play_Song:

			if(!Instr_Datas_Copied) {
				Size_Sequence_Dat = sizeof(SEQUENCE);
				Size_Instrument_Vars = sizeof(INSTRUMENT_VARS);
				if(Play_Type == Play_Song) {
					for(i = 0; i < 8; i++) {
						Clear_Sequence_Dats(&Sequences_Dats[i]);
					}
				} else {
					Clear_Sequence_Dats(&Dummy_Sequence_Dats);
				}
				Global_Volume = 1.0f;
				Cur_Ticks = 0;
				Fx_Ticks = 0;
				Arpeggio_Counter = 3;
				Cur_Seq_Play_Pos = 15;
			}
			Instr_Datas_Copied = 1;

			_asm {
				mov		edx,[DatSize]
				push	dword ptr [Bytes]
				mov		[BufferSize],edx
				shr		edx,2
				mov		edi,offset Mixing_buffer_Stereo
Mix_Loop:		test	edx,edx
				jz		End_Mix_Loop
				push	edx
				push	ebp

				// -------------------------------------------------
				// Sequence playback
				mov		ebp,offset Dummy_Sequence_Dats
				cmp		dword ptr [Play_Type],Play_Song
				jne		Song_PlayBack
				mov		ebp,offset Sequences_Dats

				// Set voice status
				mov		ecx,8
				mov		ebx,offset OnOff_Status
				lea		edx,[ebp]
Set_Voices_On_Off:				
				mov		eax,[ebx]
				test	eax,eax
				jnz		Play_Voice
				mov		dword ptr [edx + SEQUENCE.Seq_Mute_Volume],0
				jmp		Mute_Voice
Play_Voice:		mov		dword ptr [edx + SEQUENCE.Seq_Mute_Volume],0x3f800000
Mute_Voice:		add		edx,[Size_Sequence_Dat]
				add		ebx,4
				loop	Set_Voices_On_Off
Song_PlayBack:	mov		[Ptr_Sequence],ebp
				cmp		dword ptr [Play_Type],Play_Instrument
				je		Instrument_PlayBack
				mov		dword ptr [Speed_Tick],0
				dec		dword ptr [Fx_Ticks]
				jns		Speed_Limit
				push	dword ptr [Fx_Tempo_Ticks]
				pop		dword ptr [Fx_Ticks]
				inc		dword ptr [Speed_Tick]
				inc		dword ptr [Arpeggio_Counter]
				cmp		dword ptr [Arpeggio_Counter],3
				jl		Speed_Limit
				mov		dword ptr [Arpeggio_Counter],0
Speed_Limit:	dec		dword ptr [Cur_Ticks]
				jns		Update_Channels
				push	dword ptr [Tempo_Ticks]
				pop		dword ptr [Cur_Ticks]
				// Goto next position
				inc		dword ptr [Cur_Seq_Play_Pos]
				cmp		dword ptr [Cur_Seq_Play_Pos],16
				jne		Reset_Sequence_Pos
				mov		dword ptr [Cur_Seq_Play_Pos],0
				cmp		dword ptr [Play_Type],Play_Song
				jne		Reset_Pans
				call	Get_Positions_Number
				inc		dword ptr [Song_Position]
				cmp		dword ptr [Song_Position],eax
				jl		Reset_Song_Position
				xor		ebx,ebx
				test	eax,eax
				jz		Max_Restart
				mov		ebx,[Song_Restart]
				dec		ebx
				dec		eax
				cmp		ebx,eax
				jb		Max_Restart
				mov		ebx,eax
Max_Restart:	mov		[Song_Position],ebx
Reset_Song_Position:
				// Fill the sequences datas
				lea		ebx,[ebp]
				xor		esi,esi
				mov		ecx,8
Set_Sequences:	push	ecx
				push	dword ptr [Song_Position]
				push	esi
				call	Get_Position_Dat
				movzx	edx,byte ptr [eax + SONG_DAT.Volume]
				push	edx
				movzx	edx,byte ptr [eax + SONG_DAT.Transpose]
				mov		cl,dl
				and		dl,0x7f
				and		cl,0x80
				jz		Negative_Transpose
				neg		edx
Negative_Transpose:
				push	edx
				movzx	eax,word ptr [eax + SONG_DAT.Seq_Nbr]
				test	eax,eax
				jz		No_Sequence
				dec		eax
				push	eax
				call	Get_Sequence_Dat
No_Sequence:	mov		[ebx + SEQUENCE.Seq_Cur_Sequence],eax
				// Set transpose for this sequence
				pop		dword ptr [ebx + SEQUENCE.Seq_Position_Transpose]
				// Set global volume for this sequence
				pop		eax
				test	eax,eax
				jz		No_New_Volume
				dec		eax
				push	eax
				fild	dword ptr [esp]
				fmul	dword ptr [Flt001]
				pop		eax
				fstp	dword ptr [ebx + SEQUENCE.Seq_Position_Volume]
No_New_Volume:	add		ebx,[Size_Sequence_Dat]
				inc		esi
				pop		ecx
				loop	Set_Sequences
				// Set the default pans of the sequences
				lea		eax,[ebp]
				mov		ecx,8
Reset_Pan:		fld		dword ptr [Instrument_Pan]		// Reset the default panning
				fstp	dword ptr [eax + SEQUENCE.Seq_Pan]
				add		eax,[Size_Sequence_Dat]
				loop	Reset_Pan
				jmp		Reset_Sequence_Pos
Reset_Pans:		lea		eax,[ebp]
				fld		dword ptr [Instrument_Pan]		// Reset the default panning
				fstp	dword ptr [eax + SEQUENCE.Seq_Pan]
Reset_Sequence_Pos:
				mov		ecx,8
Set_Song_Datas:	push	ecx
				push	edi
				mov		dword ptr [ebp + SEQUENCE.Seq_Vol_Side],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Arpeggio],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Pitch_Bend],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Vibrato],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Vibrato_Freq],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Tone_Portamento_Speed],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Tone_Portamento_Note],0
				// Set the effects
				mov		edi,[ebp + SEQUENCE.Seq_Cur_Sequence]
				test	edi,edi
				jz		No_New_Seq_Panning
				mov		esi,[Cur_Seq_Play_Pos]
				movzx	ebx,byte ptr [edi + esi + SEQUENCE_DAT.Fx]
				movzx	eax,byte ptr [edi + esi + SEQUENCE_DAT.FxDatas]
				// Arpeggio
				dec		ebx
				js		Fx_NoFx
				jnz		Fx_Arpeggio
				test	eax,eax
				jnz		Fx_Old_Arpeggio
				mov		eax,[ebp + SEQUENCE.Seq_Old_Arpeggio]
Fx_Old_Arpeggio:
				mov		[ebp + SEQUENCE.Seq_Arpeggio],eax
				mov		[ebp + SEQUENCE.Seq_Old_Arpeggio],eax
Fx_Arpeggio:	// Pitch bend up
				dec		ebx
				jnz		Fx_Pitch_Bend_Up
				test	eax,eax
				jnz		Fx_Old_Pitch_Bend_Up
				mov		eax,[ebp + SEQUENCE.Seq_Old_Pitch_Bend_Up]
Fx_Old_Pitch_Bend_Up:
				mov		[ebp + SEQUENCE.Seq_Pitch_Bend],eax
				mov		[ebp + SEQUENCE.Seq_Old_Pitch_Bend_Up],eax
Fx_Pitch_Bend_Up: // Pitch bend down
				dec		ebx
				jnz		Fx_Pitch_Bend_Down
				neg		eax
				test	eax,eax
				jnz		Fx_Old_Pitch_Bend_Down
				mov		eax,[ebp + SEQUENCE.Seq_Old_Pitch_Bend_Down]
Fx_Old_Pitch_Bend_Down:
				mov		[ebp + SEQUENCE.Seq_Pitch_Bend],eax
				mov		[ebp + SEQUENCE.Seq_Old_Pitch_Bend_Down],eax
Fx_Pitch_Bend_Down:
				// Vibrato
				dec		ebx
				jnz		Fx_Vibrato
				test	eax,eax
				jnz		Fx_Old_Vibrato
				mov		eax,[ebp + SEQUENCE.Seq_Old_Vibrato]
Fx_Old_Vibrato:	mov		[ebp + SEQUENCE.Seq_Vibrato],eax
				mov		[ebp + SEQUENCE.Seq_Old_Vibrato],eax
Fx_Vibrato:		// Volume slide up
				dec		ebx
				jnz		Fx_Vol_Slide_Up
				test	eax,eax
				jnz		Fx_Old_Vol_Slide_Up
				mov		eax,[ebp + SEQUENCE.Seq_Old_Vol_Side_Up]
Fx_Old_Vol_Slide_Up:
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt001]
				fstp	dword ptr [ebp + SEQUENCE.Seq_Vol_Side]
				mov		[ebp + SEQUENCE.Seq_Old_Vol_Side_Up],eax
Fx_Vol_Slide_Up: // Volume slide down
				dec		ebx
				jnz		Fx_Vol_Slide_Down
				test	eax,eax
				jnz		Fx_Old_Vol_Slide_Down
				mov		eax,[ebp + SEQUENCE.Seq_Old_Vol_Side_Down]
Fx_Old_Vol_Slide_Down:
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt001]
				fchs
				fstp	dword ptr [ebp + SEQUENCE.Seq_Vol_Side]
				mov		[ebp + SEQUENCE.Seq_Old_Vol_Side_Down],eax
Fx_Vol_Slide_Down:
				// Modify Tempo
				dec		ebx
				jnz		Fx_Set_Tempo
				mov		[Tempo],eax
				push	eax
				call	Get_Mix_Buffer_Len
				mov		[Tempo_Ticks],eax
				push	ebx
				push	esi
				push	edi
				push	ebp
				push	dword ptr [Tempo]
				call	Display_Tempo
				pop		ebp
				pop		edi
				pop		esi
				pop		ebx
Fx_Set_Tempo:	// Modify Misc. FX Tempo
				dec		ebx
				jnz		Fx_Set_Misc_FX_Tempo
				mov		[Fx_Tempo],eax
				push	eax
				call	Get_Mix_Buffer_Len
				mov		[Fx_Tempo_Ticks],eax
				push	ebx
				push	esi
				push	edi
				push	ebp
				push	dword ptr [Fx_Tempo]
				call	Display_Fx_Tempo
				pop		ebp
				pop		edi
				pop		esi
				pop		ebx
Fx_Set_Misc_FX_Tempo:
				// Set global volume
				dec		ebx
				jnz		Fx_Set_Global_Volume
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt001]
				call	Clamp_Value
				fstp	dword ptr [Global_Volume]
Fx_Set_Global_Volume:
				// Tone portamento
				dec		ebx
				jnz		Fx_Tone_Portamento
				movzx	edx,byte ptr [edi + esi + SEQUENCE_DAT.Note]
				test	eax,eax
				jnz		Fx_Old_Tone_Portamento_Speed
				mov		eax,[ebp + SEQUENCE.Seq_Old_Tone_Portamento_Speed]
Fx_Old_Tone_Portamento_Speed:
				dec		edx							// Must begins at 0
				jns		Fx_Old_Tone_Portamento_Note
				mov		edx,[ebp + SEQUENCE.Seq_Old_Tone_Portamento_Note]
Fx_Old_Tone_Portamento_Note:
				mov		[ebp + SEQUENCE.Seq_Tone_Portamento_Speed],eax
				mov		[ebp + SEQUENCE.Seq_Old_Tone_Portamento_Speed],eax
				mov		[ebp + SEQUENCE.Seq_Tone_Portamento_Note],edx
				mov		[ebp + SEQUENCE.Seq_Old_Tone_Portamento_Note],edx
Fx_Tone_Portamento:
				// External program control
				dec		ebx
				jnz		Fx_Extern_Control
				push	esi
				push	edi
				mov		ebx,[Synth_CallBack]
				test	ebx,ebx
				jz		No_Synchro
				push	eax
				call	ebx
No_Synchro:		pop		edi
				pop		esi
Fx_Extern_Control:
Fx_NoFx:		// Set the instrument
				movzx	ebx,word ptr [edi + SEQUENCE_DAT.Instrument + esi * 2]
				dec		ebx
				js		No_New_Instrument
				push	ebx
				call	Get_Instrument
				mov		[ebp + SEQUENCE.Seq_Current_Instrument],eax
				push	0
				push	dword ptr [ebp + SEQUENCE.Seq_Current_Instrument]
				call	Get_Instrument_Dat
				mov		[ebp + SEQUENCE.Seq_Current_Instrument_Dat],eax
				test	eax,eax
				jz		No_New_Instrument
				push	dword ptr [ebp + SEQUENCE.Seq_Current_Instrument]
				call	Get_Instrument_Comb
				mov		[ebp+SEQUENCE.Seq_Instrument_NbrComb],eax
				fld1
				fstp	dword ptr [ebp + SEQUENCE.Seq_Volume]
No_New_Instrument:
				// Set the note
				mov		esi,[Cur_Seq_Play_Pos]
				cmp		dword ptr [ebp + SEQUENCE.Seq_Tone_Portamento_Speed],0
				jne		No_New_Note
				movzx	ebx,byte ptr [edi + esi + SEQUENCE_DAT.Note]
				dec		ebx							// Must begins at 0
				js		No_New_Note
				add		ebx,[ebp + SEQUENCE.Seq_Position_Transpose]
				mov		[ebp + SEQUENCE.Seq_Cur_Note],ebx
				mov		[ebp + SEQUENCE.Seq_Cur_Freq],ebx
				mov		dword ptr [ebp + SEQUENCE.Seq_Pitch_Bend_Freq],0
				mov		eax,[ebp + SEQUENCE.Seq_Current_Instrument]
				test	eax,eax
				jz		No_New_Note
				mov		ecx,[ebp + SEQUENCE.Seq_Instrument_NbrComb]
				lea		eax,[ebp + SEQUENCE.Seq_Cur_Instrument_Vars]
Clear_Instrument_Dats:
				mov		dword ptr [eax + INSTRUMENT_VARS.Instrument_Vol], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.Instrument_Tick], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.wrn1], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.wrn2], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.wrpos], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.FPos], 0
				mov		dword ptr [eax + INSTRUMENT_VARS.Oscillator],0
				mov		dword ptr [eax + INSTRUMENT_VARS.Frac_Lo_Pos],0
				add		eax,[Size_Instrument_Vars]
				loop	Clear_Instrument_Dats
No_New_Note:	// Set the volume
				movzx	ebx,byte ptr [edi + esi + SEQUENCE_DAT.Volume]
				dec		ebx
				js		No_New_Seq_Volume
				push	ebx
				fild	dword ptr [esp]
				pop		ebx
				fmul	dword ptr [Flt001]
				fstp	dword ptr [ebp + SEQUENCE.Seq_Volume]
No_New_Seq_Volume:
				// Set the panning
				movzx	ebx,byte ptr [edi + esi + SEQUENCE_DAT.Panning]
				dec		ebx
				js		No_New_Seq_Panning
				push	ebx
				fild	dword ptr [esp]
				pop		ebx
				fmul	dword ptr [Flt001]
				fstp	dword ptr [ebp + SEQUENCE.Seq_Pan]
No_New_Seq_Panning:
				pop		edi
				pop		ecx
				cmp		dword ptr [Play_Type],Play_Song
				jne		Update_Channels
				add		ebp,[Size_Sequence_Dat]
				dec		ecx
				jnz		Set_Song_Datas
				jmp		Update_Channels
Instrument_PlayBack:
				// -------------------------------------------------
				// Single instrument playback
				mov		ebx,[Current_Instr_Note]	// Default note for instruments playback
				dec		ebx
				mov		[Dummy_Sequence_Dats.Seq_Cur_Freq],ebx
				// Set sample
				push	0
				push	dword ptr [Current_Instrument]
				call	Get_Instrument_Dat
				mov		[Dummy_Sequence_Dats.Seq_Current_Instrument_Dat],eax
				push	dword ptr [Current_Instrument]
				call	Get_Instrument_Comb
				mov		[Dummy_Sequence_Dats.Seq_Instrument_NbrComb],eax
Update_Channels:// -------------------------------------------------
				// Process inbetween effects
				cmp		dword ptr [Play_Type],Play_Instrument
				je		Done_With_Fx
				mov		ebp,[Ptr_Sequence]
				mov		ecx,8
Set_Song_Fx:	push	ecx
				cmp		dword ptr [ebp + SEQUENCE.Seq_Pitch_Bend],0
				je		No_PitchBend
				cmp		dword ptr [Speed_Tick],0
				je		No_PitchBend
				mov		eax,[ebp + SEQUENCE.Seq_Pitch_Bend_Freq]
				add		eax,[ebp + SEQUENCE.Seq_Pitch_Bend]
				mov		[ebp + SEQUENCE.Seq_Pitch_Bend_Freq],eax
No_PitchBend:	cmp		dword ptr [ebp + SEQUENCE.Seq_Vibrato],0
				je		No_Vibrato
				cmp		dword ptr [Speed_Tick],0
				je		No_Vibrato
				mov		eax,[ebp + SEQUENCE.Seq_Vibrato]
				mov		ebx,eax
				shr		ebx,4			// Speed
				and		eax,0fh			// Depth
				push	eax
				fild	dword ptr [esp]
				fadd	st(0),st(0)
				fadd	st(0),st(0)
				fadd	st(0),st(0)
				fadd	st(0),st(0)
				pop		eax
				push	dword ptr [ebp + SEQUENCE.Seq_Vibrato_Tick]
				fild	dword ptr [esp]
				fmul	dword ptr [Flt005]
				fsin
				fmulp	st(1),st(0)
				fistp	dword ptr [esp]
				pop		eax
				mov		[ebp + SEQUENCE.Seq_Vibrato_Freq],eax
				add		[ebp + SEQUENCE.Seq_Vibrato_Tick],ebx
No_Vibrato:		cmp		dword ptr [ebp + SEQUENCE.Seq_Arpeggio],0
				je		No_Arpeggio
				mov		eax,[Arpeggio_Counter]
				xor		edx,edx
				mov		ebx,3
				div		ebx
				dec		edx
				jz		Do_Arpeggio1
				dec		edx
				jz		Do_Arpeggio2
				xor		edx,edx
				jmp		Do_Arpeggio
Do_Arpeggio1:	mov		edx,[ebp + SEQUENCE.Seq_Arpeggio]
				shr		edx,4
				jmp		Do_Arpeggio
Do_Arpeggio2:	mov		edx,[ebp + SEQUENCE.Seq_Arpeggio]
				and		edx,0fh
Do_Arpeggio:	add		edx,[ebp + SEQUENCE.Seq_Cur_Note]
				mov		[ebp + SEQUENCE.Seq_Cur_Freq],edx
No_Arpeggio:	cmp		dword ptr [ebp + SEQUENCE.Seq_Tone_Portamento_Speed],0
				je		No_TonePortamento
				cmp		dword ptr [Speed_Tick],0
				je		No_TonePortamento
				mov		ebx,[ebp + SEQUENCE.Seq_Pitch_Bend_Freq]
				mov		ecx,[ebp + SEQUENCE.Seq_Tone_Portamento_Speed]
				mov		edx,[ebp + SEQUENCE.Seq_Tone_Portamento_Note]
				mov		eax,[Freqs_Table + edx * 4]
				cmp		eax,[ebp + SEQUENCE.Seq_Current_Freq]
				je		Portamento_Stop
				ja		Portamento_Up
				sub		ebx,ecx
				cmp		eax,ebx
				ja		Portamento_Stop
				jmp		Set_Portamento_Freq
Portamento_Up:	add		ebx,ecx
				cmp		eax,ebx
				jb		Portamento_Stop
				jmp		Set_Portamento_Freq
Portamento_Stop:		mov		[ebp + SEQUENCE.Seq_Cur_Freq],edx
				xor		ebx,ebx
Set_Portamento_Freq:		mov		[ebp + SEQUENCE.Seq_Pitch_Bend_Freq],ebx
No_TonePortamento:		pop		ecx
				cmp		dword ptr [Play_Type],Play_Song
				jne		Done_With_Fx
				add		ebp,[Size_Sequence_Dat]
				dec		ecx
				jnz		Set_Song_Fx
Done_With_Fx:	// -------------------------
				// Handle waves combinations
				mov		ebp,[Ptr_Sequence]
				mov		ecx,8
Set_Song_Instruments:		push	ecx
				push	edi
				lea		eax,[ebp+SEQUENCE.Seq_Cur_Instrument_Vars]
				mov		[Instr_Dat_Vars],eax
				mov		dword ptr [ebp + SEQUENCE.Sound_Datas],0
				mov		dword ptr [Instr_Dat_Number], 0
				mov		ebx,[ebp + SEQUENCE.Seq_Current_Instrument_Dat]
				// Load number of combinators
				mov		ecx,[ebp + SEQUENCE.Seq_Instrument_NbrComb]
Load_Instrument:		test	ebx,ebx
				jz		No_Instrument
				push	ecx
				mov		edi,[Instr_Dat_Vars]
				mov		ecx,[ebp + SEQUENCE.Seq_Cur_Freq]
				mov		dword ptr [ebp + SEQUENCE.Seq_Freq_Hi_Speed],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Freq_Lo_Speed],0
				test	ecx,ecx
				js		No_Freq
				cmp		ecx,MAXNOTE
				jge		No_Freq
				mov		eax,[Freqs_Table + ecx * 4]
				add		eax,[ebp + SEQUENCE.Seq_Vibrato_Freq]
				add		eax,[ebp + SEQUENCE.Seq_Pitch_Bend_Freq]
				movzx	edx,byte ptr [ebx + INSTRUMENT_DAT.Waveform]
				cmp		dl,INS_WF_CUST										// Use phase shift as an absolute value
				je		Custom_Value
				movsx	edx,word ptr [ebx + INSTRUMENT_DAT.PhaseShift]
				add		eax,edx
Custom_Value:	cmp		eax,0417h
				jg		Min_Frequency
				mov		eax,0417h
Min_Frequency:	cmp		eax,0F6F11h
				jl		Max_Frequency
				mov		eax,0F6F11h
Max_Frequency:	mov		[ebp + SEQUENCE.Seq_Current_Freq],eax
				xor		edx,edx
				div		dword ptr [Max_Buffer]
				mov		[ebp + SEQUENCE.Seq_Freq_Hi_Speed],eax
				xor		eax,eax
				div		dword ptr [Max_Buffer]
				mov		[ebp + SEQUENCE.Seq_Freq_Lo_Speed],eax
No_Freq:		mov		edx,[edi + INSTRUMENT_VARS.Frac_Lo_Pos]
				mov		esi,[edi + INSTRUMENT_VARS.Oscillator]
				add		edx,[ebp + SEQUENCE.Seq_Freq_Lo_Speed]
				adc		esi,[ebp + SEQUENCE.Seq_Freq_Hi_Speed]
				mov		[edi + INSTRUMENT_VARS.Frac_Lo_Pos],edx
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				mov		[edi + INSTRUMENT_VARS.Oscillator],esi
				fild	dword ptr [edi + INSTRUMENT_VARS.Oscillator]
				fmul	dword ptr [Flt01]
				movzx	edx,byte ptr [ebx + INSTRUMENT_DAT.Waveform]
				dec		dl
				jnz		Set_Ins_Sin
				fsin
Set_Ins_Sin:	dec		dl
				jnz		Set_Ins_Sqr
				fsin
				fld		st(0)
				push	eax
				fistp	dword ptr [esp]
				fild	dword ptr [esp]
				fxch	st(1)
				cmp		dword ptr [esp],0
				jne		Sqr_Zero
				ftst
				fstsw	ax
				sahf
				fstp	st(0)
				fstp	st(0)
				fld1
				ja		Sqr_Neg
				fchs
Sqr_Neg:		fld		st(0)
Sqr_Zero:		fstp	st(0)
				pop		eax
Set_Ins_Sqr:	dec		dl
				jnz		Set_Ins_Saw
				fld		dword ptr [FltPI2]
				fxch	st(1)
FModf_Saw:		fprem
				fstsw	ax
				sahf
				jp		FModf_Saw
				fstp	st(1)
				fdiv	dword ptr [FltPI]
				fsub	dword ptr [Flt1]
Set_Ins_Saw:	dec		dl
				jnz		Set_Ins_Rnd
				fstp	dword ptr [edi + INSTRUMENT_VARS.FPos]
				cmp		dword ptr [edi + INSTRUMENT_VARS.wrpos],0
				jne		Reset_Noise
				call	Wave_WhiteNoise
				fstp	dword ptr [edi + INSTRUMENT_VARS.wrn1]
				call	Wave_WhiteNoise
				fstp	dword ptr [edi + INSTRUMENT_VARS.wrn2]
				mov		eax,[edi + INSTRUMENT_VARS.FPos]
				mov		[edi + INSTRUMENT_VARS.wrpos],eax
Reset_Noise:	fld		dword ptr [edi + INSTRUMENT_VARS.wrpos]
				fadd	dword ptr [FltPI2]
				fcom	dword ptr [edi + INSTRUMENT_VARS.FPos]
				fnstsw	ax
				sahf
				jae		Done_Rnd
				fstp	dword ptr [edi + INSTRUMENT_VARS.wrpos]
				mov		eax,[edi + INSTRUMENT_VARS.wrn2]
				mov		[edi + INSTRUMENT_VARS.wrn1],eax
				call	Wave_WhiteNoise
				fstp	dword ptr [edi + INSTRUMENT_VARS.wrn2]
				jmp		Reset_Noise
Done_Rnd:		fstp	st(0)
				fld		dword ptr [edi + INSTRUMENT_VARS.FPos]
				fsub	dword ptr [edi + INSTRUMENT_VARS.wrpos]
				fmul	dword ptr [edi + INSTRUMENT_VARS.wrn2]
				fld		dword ptr [FltPI2]
				fadd	dword ptr [edi + INSTRUMENT_VARS.wrpos]
				fsub	dword ptr [edi + INSTRUMENT_VARS.FPos]
				fmul	dword ptr [edi + INSTRUMENT_VARS.wrn1]
				faddp	st(1),st(0)
				fmul	dword ptr [Flt05]
Set_Ins_Rnd:	dec		dl
				jnz		Set_Ins_Tri
				fld		dword ptr [FltPI2]
				fxch	st(1)
FModf_Tri:		fprem
				fstsw	ax
				sahf
				jp		FModf_Tri
				fstp	st(1)
				fcom	dword ptr [FltPI]
				fnstsw	ax
				sahf
				fld1
				ja		Tri_Desc
				fchs
Tri_Desc:		fxch	st(1)
				fcom	dword ptr [FltPI2]
				fnstsw	ax
				sahf
				jb		Tri_Ramp_Up
				fsub	dword ptr [FltPI2]
				fld		dword ptr [FltPI2]
				fsubrp	st(1),st(0)
Tri_Ramp_Up:	fdiv	dword ptr [FltPI2]
				fmulp	st(1),st(0)
				fadd	dword ptr [Flt025]
				fcom	dword ptr [Flt05]
				fnstsw	ax
				sahf
				jb		Tri_Invert
				fsub	dword ptr [Flt1]
Tri_Invert:		fadd	st(0),st(0)
				fadd	st(0),st(0)
Set_Ins_Tri:	dec		dl
				jnz		Set_Ins_Cust
				fstp	st(0)
				movzx	eax,byte ptr [ebx+INSTRUMENT_DAT.PhaseShift]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				cmp		word ptr [ebx+INSTRUMENT_DAT.PhaseShift],0
				jg		Negative_Shift
				fchs
Negative_Shift:	fmul	dword ptr [Flt001]
Set_Ins_Cust:	push	ebp
				mov		ebp,[Instr_Dat_Vars]
				// Handle ADSR
				push	edi
				movzx	edi,byte ptr [ebx+INSTRUMENT_DAT.Attack_Vol]
				movzx	ecx,word ptr [ebx+INSTRUMENT_DAT.Attack]
				xor		edx,edx
				movsx	eax,byte ptr [ebx+INSTRUMENT_DAT.Attack_Vol_Delta]
				mov		esi,ecx
				test	ecx,ecx
				jz		Load_Decay
				add		edx,ecx
				cmp		dword ptr [ebp + INSTRUMENT_VARS.Instrument_Tick],edx
				jb		Load_Decay
				movzx	edi,byte ptr [ebx + INSTRUMENT_DAT.Decay_Vol]
				movzx	esi,word ptr [ebx + INSTRUMENT_DAT.Decay]
				movsx	eax,byte ptr [ebx + INSTRUMENT_DAT.Decay_Vol_Delta]
Load_Decay:		movzx	ecx,word ptr [ebx + INSTRUMENT_DAT.Decay]
				test	ecx,ecx
				jz		Load_Sustain
				add		edx,ecx
				cmp		[ebp + INSTRUMENT_VARS.Instrument_Tick],edx
				jb		Load_Sustain
				movzx	edi,byte ptr [ebx + INSTRUMENT_DAT.Sustain_Vol]
				movzx	esi,word ptr [ebx + INSTRUMENT_DAT.Sustain]
				movsx	eax,byte ptr [ebx + INSTRUMENT_DAT.Sustain_Vol_Delta]
Load_Sustain:	movzx	ecx,word ptr [ebx + INSTRUMENT_DAT.Sustain]
				test	ecx,ecx
				jz		Load_Release
				add		edx,ecx
				cmp		dword ptr [ebp + INSTRUMENT_VARS.Instrument_Tick],edx
				jb		Load_Release
				movzx	edi,byte ptr [ebx +INSTRUMENT_DAT.Release_Vol]
				movzx	esi,word ptr [ebx + INSTRUMENT_DAT.Release]
				movsx	eax,byte ptr [ebx + INSTRUMENT_DAT.Release_Vol_Delta]
Load_Release:	test	esi,esi
				jz		Done_Ramping
				test	eax,eax
				jz		Done_Ramping
				push	eax
				jg		Vol_Ramp_Up
				// Ramping down
				push	edi						// Dest volume
				fild	dword ptr [esp]
				pop		edi
				fmul	dword ptr [Flt001]
				fcomp	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
				fnstsw	ax
				sahf
				jb		Volume_Max
				jmp		Volume_Equal
				// Ramping up
Vol_Ramp_Up:	push	edi
				fild	dword ptr [esp]
				pop		edi
				fmul	dword ptr [Flt001]
				fcomp	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
				fnstsw	ax
				sahf
				jz		Volume_Equal
				ja		Volume_Max
Volume_Equal:	// Set the Dest volume
				mov		[esp],edi
				fild	dword ptr [esp]
				pop		edi
				fmul	dword ptr [Flt001]
				fstp	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
				jmp		Done_Ramping
Volume_Max:		fild	dword ptr [esp]			// Delta volume
				pop		eax
				push	esi
				fild	dword ptr [esp]			// Ramping time
				pop		esi
				fdivp	st(1),st(0)
				fmul	dword ptr [Flt001]
				fadd	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
				call	Clamp_Value
				fstp	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
Done_Ramping:	// Set the volume of the sample according to the ADSR
				fmul	dword ptr [ebp + INSTRUMENT_VARS.Instrument_Vol]
				pop		edi
				// Handle operators
				mov		dl,byte ptr [ebx + INSTRUMENT_DAT.Operator]
				dec		dl
				jnz		Set_Ins_Op_Add
				faddp	st(1),st(0)
Set_Ins_Op_Add:	dec		dl
				jnz		Set_Ins_Op_Sub
				fsubp	st(1),st(0)
Set_Ins_Op_Sub:	dec		dl
				jnz		Set_Ins_Op_Mul
				fmulp	st(1),st(0)
Set_Ins_Op_Mul:	dec		dl
				jnz		Set_Ins_Op_Div
				fdivp	st(1),st(0)
Set_Ins_Op_Div:	inc		dword ptr [ebp + INSTRUMENT_VARS.Instrument_Tick]
				pop		ebp
				// Store oscillator value now
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
				// Next instrument's datas
				inc		dword ptr [Instr_Dat_Number]
				push	dword ptr [Instr_Dat_Number]
				cmp		dword ptr [Play_Type],Play_Instrument
				je		Sequence_Instrument
				// Next datas of the instrument of the sequence
				push	dword ptr [ebp + SEQUENCE.Seq_Current_Instrument]
				call	Get_Instrument_Dat
				jmp		Next_Instrument
Sequence_Instrument:
				// Next datas of the current instrument
				push	dword ptr [Current_Instrument]
				call	Get_Instrument_Dat
Next_Instrument:
				mov		ebx,eax
				mov		eax,[Size_Instrument_Vars]
				add		[Instr_Dat_Vars],eax
				pop		ecx
				dec		ecx
				jnz		Load_Instrument
No_Instrument:	pop		edi
				cmp		dword ptr [Play_Type],Play_Instrument
				je		No_Process_Filters
				// Set channel volume
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				fld		dword ptr [ebp + SEQUENCE.Seq_Volume]
				cmp		dword ptr [Speed_Tick],0
				je		No_Vol_Slide
				fadd	dword ptr [ebp + SEQUENCE.Seq_Vol_Side]
				call	Clamp_Value
				fst		dword ptr [ebp + SEQUENCE.Seq_Volume]
No_Vol_Slide:	fmulp	st(1),st(0)
				// Set song position volume
				fmul	dword ptr [ebp + SEQUENCE.Seq_Position_Volume]
				fmul	dword ptr [Global_Volume]
				fmul	dword ptr [ebp + SEQUENCE.Seq_Mute_Volume]
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
				// Reset DSP effects parameters
				mov		edx,[ebp+SEQUENCE.Seq_Cur_Sequence]
				test	edx,edx
				jz		No_DSP_Params
				mov		esi,[Cur_Seq_Play_Pos]
				mov		bl,byte ptr [edx + esi + SEQUENCE_DAT.DspMask]
				test	bl,1
				jz		Switch_Lo_Pass_Off
				mov		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param1],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param2],0
Switch_Lo_Pass_Off:
				test	bl,2
				jz		Switch_Hi_Pass_Off
				mov		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param1],0
				mov		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param2],0
Switch_Hi_Pass_Off:
				test	bl,4
				jz		Switch_VCF_Pass_Off
				mov		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param1],0
				mov		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param2],0
Switch_VCF_Pass_Off:
				test	bl,8
				jz		Switch_Delay_Off
				push	0
				push	0
				push	0
				push	ebp
				call	Set_Delay
				mov		dword ptr [ebp + SEQUENCE.Seq_Delay_Cursor],0
Switch_Delay_Off:
				test	bl,16
				jz		Switch_Disto_Off
				mov		dword ptr [ebp + SEQUENCE.Disto_Param],0
Switch_Disto_Off:
				test	bl,32
				jz		Switch_Formant_Off
				mov		dword ptr [ebp + SEQUENCE.Formant_Param1],0
Switch_Formant_Off:
				test	bl,64
				jz		Switch_Reverb_Off
				push	0
				push	0
				push	0
				push	ebp
				call	Set_Reverb
				mov		dword ptr [ebp + SEQUENCE.Seq_Reverb_Cursor],0
Switch_Reverb_Off:
				// Process DSP effects
				mov		bl,byte ptr [edx + esi + SEQUENCE_DAT.Dsp]
				dec		bl
				js		No_DSP_Params
				jnz		No_Lo_Pass_Filter
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas3]
				mov		ah,[edx + esi + SEQUENCE_DAT.DspDatas2]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fstp	dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param2]
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas1]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt01]
				fstp	dword ptr [ebp+SEQUENCE.Seq_Lo_Pass_Param1]
				jmp		No_DSP_Params
No_Lo_Pass_Filter:
				dec		bl
				jnz		No_Hi_Pass_Filter
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas3]
				mov		ah,[edx + esi + SEQUENCE_DAT.DspDatas2]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fstp	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param2]
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas1]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt01]
				fstp	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param1]
				jmp		No_DSP_Params
No_Hi_Pass_Filter:
				dec		bl
				jnz		No_VCF_Pass_Filter
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas3]
				mov		ah,[edx + esi + SEQUENCE_DAT.DspDatas2]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt00001]
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param2]
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas1]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt001]
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param1]
				jmp		No_DSP_Params
No_VCF_Pass_Filter:
				// Delay is processed below
				dec		bl
				dec		bl
				jnz		No_Disto_Filter
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas3]
				mov		ah,[edx + esi + SEQUENCE_DAT.DspDatas2]
				push	eax
				fild	dword ptr [esp]
				pop		eax
				fmul	dword ptr [Flt01]
				fstp	dword ptr [ebp + SEQUENCE.Disto_Param]
				jmp		No_DSP_Params
No_Disto_Filter:
				dec		bl
				jnz		No_Formant_Filter
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas1]
				cmp		eax,5
				jb		Max_Formant_Filter
				mov		eax,5
Max_Formant_Filter:
				mov		[ebp + SEQUENCE.Formant_Param1],eax
No_Formant_Filter:
				// Reverb is processed below
No_DSP_Params:	cmp		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param1],0
				je		No_Process_Lo_Pass
				cmp		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param2],0
				je		No_Process_Lo_Pass
				fldpi
				fmul	dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param2]
				fmul	dword ptr [Flt0000023]
				fptan
				fstp	st(0)
				mov		edx,dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_In1]
				mov		ecx,dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Out1]
				fld1
				fld		st(0)
				fdivrp	st(2),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Param1]
				fmul	st(0),st(2)
				fxch	st(2)
				fmul	st(0),st(0)
				fld		st(1)
				fadd	st(0),st(3)
				fadd	st(0),st(1)
				fdivr	st(0),st(2)
				fld		st(0)
				fadd	st(0),st(0)
				fld		st(3)
				fsub	st(0),st(3)
				fadd	st(0),st(0)
				fmul	st(0),st(2)
				fxch	st(4)
				fsubrp	st(5),st(0)
				fxch	st(2)
				faddp	st(4),st(0)
				fmul	st(3),st(0)
				fld		st(0)
				fmul	dword ptr [ebp + SEQUENCE.Sound_Datas]
				fxch	st(2)
				fmul	dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_In1]
				faddp	st(2),st(0)
				fmul	dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_In2]
				faddp	st(1),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Out1]
				fchs
				fmulp	st(2),st(0)
				faddp	st(1),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Out2]
				fchs
				fmulp	st(2),st(0)
				faddp	st(1),st(0)
				mov		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_In2],edx
				fld		st(0)
				mov		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Out2],ecx
				push	dword ptr [ebp + SEQUENCE.Sound_Datas]
				pop		dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_In1]
				fxch	st(1)
				fstp	dword ptr [ebp + SEQUENCE.Seq_Lo_Pass_Out1]
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
No_Process_Lo_Pass:
				cmp		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param1],0
				je		No_Process_Hi_Pass
				cmp		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param2],0
				je		No_Process_Hi_Pass
				fldpi
				fmul	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param2]
				fmul	dword ptr [Flt0000023]
				fptan
				fstp	st(0)
				fld		st(0)
				fmul	st(0),st(0)
				fld1
				fld		st(0)
				fadd	st(0),st(2)
				fld		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Param1]
				fmulp	st(4),st(0)
				fld		st(0)
				fadd	st(0),st(4)
				fdivp	st(2),st(0)
				fxch	st(2)
				fadd	dword ptr [Fltm1]
				fld		dword ptr [Fltm2]
				fmul	st(0),st(2)
				fxch	st(1)
				fadd	st(0),st(0)
				fmul	st(0),st(2)
				fxch	st(4)
				fsubp	st(3),st(0)
				fxch	st(1)
				fmul	st(2),st(0)
				mov		edx,[ebp + SEQUENCE.Seq_Hi_Pass_In1]
				mov		ecx,[ebp + SEQUENCE.Seq_Hi_Pass_Out1]
				fld		st(0)
				fmul	dword ptr [ebp + SEQUENCE.Sound_Datas]
				fxch	st(2)
				fmul	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_In1]
				faddp	st(2),st(0)
				fmul	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_In2]
				faddp	st(1),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Out1]
				fchs
				fmulp	st(3),st(0)
				faddp	st(2),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Out2]
				fchs
				fmulp	st(1),st(0)
				faddp	st(1),st(0)
				mov		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_In2],edx
				fld		st(0)
				mov		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Out2],ecx
				push	dword ptr [ebp + SEQUENCE.Sound_Datas]
				pop		dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_In1]
				fxch	st(1)
				fstp	dword ptr [ebp + SEQUENCE.Seq_Hi_Pass_Out1]
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
No_Process_Hi_Pass:
				cmp		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param1],0
				je		No_Process_VCF_Pass
				cmp		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param2],0
				je		No_Process_VCF_Pass
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param2]
				fmul	dword ptr [Flt116]
				fld1
				fld		st(1)
				fmul	st(0),st(0)
				fld		dword ptr [Flt015]
				fmul	st(0),st(3)
				fmul	st(0),st(3)
				fsubr	st(0),st(2)
				fmul	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Param1]
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out4]
				fld		st(0)
				fchs
				fmulp	st(2),st(0)
				fxch	st(1)
				fadd	dword ptr [ebp + SEQUENCE.Sound_Datas]
				fxch	st(3)
				fsubrp	st(4),st(0)
				fld		dword ptr [Flt0350130]
				fmul	st(0),st(2)
				fmulp	st(2),st(0)
				fxch	st(1)
				fmulp	st(2),st(0)
				fld		st(2)
				fld		dword ptr [Flt03]
				fld		st(0)
				fld		st(5)
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In1]
				fmulp	st(2),st(0)
				fxch	st(1)
				fadd	st(0),st(5)
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out1]
				fmulp	st(4),st(0)
				faddp	st(3),st(0)
				fxch	st(2) 
				fst		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out1]
				fld		st(1)
				fxch	st(5)
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In1]
				fld		st(5)
				fxch	st(4)
				fmulp	st(6),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In2]
				fmulp	st(5),st(0)
				fadd	st(4),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out2]
				fmulp	st(3),st(0)
				fxch	st(2)
				faddp	st(4),st(0)
				fxch	st(3)
				fst		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out2]
				fld		st(3)
				fxch	st(2)
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In2]
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In3]
				fmulp	st(2),st(0)
				fadd	st(1),st(0)
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out3]
				fmulp	st(3),st(0)
				fxch	st(2)
				faddp	st(1),st(0)
				fst		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out3]
				fxch	st(1)
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In3]
				fld		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In4]
				fmulp	st(2),st(0)
				fadd	st(1),st(0)
				fxch	st(2)
				faddp	st(1),st(0)
				fst		dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_Out4]
				fxch	st(1)
				fstp	dword ptr [ebp + SEQUENCE.Seq_VCF_Pass_In4]
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
No_Process_VCF_Pass:
				cmp		dword ptr [ebp + SEQUENCE.Disto_Param],0
				je		No_Process_Disto
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				fabs
				fst		qword ptr [ebp + SEQUENCE.Disto_Abs_Input]
				fld1
				fld		dword ptr [ebp + SEQUENCE.Disto_Param]
				fld		st(0)
				fadd	qword ptr [ebp + SEQUENCE.Disto_Abs_Input]
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				fmul	st(1),st(0)
				fmul	st(0),st(0)
				fadd	st(0),st(3)
				fld		dword ptr [Fltm1]
				faddp	st(3),st(0)
				fxch	st(4)
				fmulp	st(2),st(0)
				fxch	st(1)
				faddp	st(3),st(0)
				fxch	st(1)
				fdivrp	st(2),st(0)
				fmulp	st(1),st(0)
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
No_Process_Disto:
				mov		eax,[ebp + SEQUENCE.Formant_Param1]
				test	eax,eax
				jz		No_Process_Formant
				dec		eax
				imul	eax,eax,(11 * 8)
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 5)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 6)]
				fld		qword ptr [eax + coeff]
				fmul	dword ptr [ebp + SEQUENCE.Sound_Datas]
				fld		qword ptr [eax + coeff + (8 * 1)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem]
				fld		qword ptr [eax + coeff + (8 * 2)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 1)]
				fld		qword ptr [eax + coeff + (8 * 3)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 2)]
				fld		qword ptr [eax + coeff + (8 * 4)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 3)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 7)]
				fxch	st(6)
				fst		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 7)]
				fxch	st(7)
				fst		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 6)]
				fxch	st(4)
				faddp	st(5),st(0)
				fxch	st(2)
				faddp	st(4),st(0)
				faddp	st(3),st(0)
				faddp	st(2),st(0)
				fld		qword ptr [eax + coeff + (8 * 5)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 4)]
				fld		qword ptr [eax + coeff + (8 * 6)]
				fmulp	st(2),st(0)
				faddp	st(1),st(0)
				faddp	st(1),st(0)
				fld		qword ptr [eax + coeff + (8 * 7)]
				fmulp	st(3),st(0)
				fld		qword ptr [eax + coeff + (8 * 8)]
				fmul	st(0),st(2)
				faddp	st(3),st(0)
				fld		qword ptr [eax + coeff + (8 * 9)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 8)]
				fld		qword ptr [eax + coeff + (8 * 10)]
				fmul	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 9)]
				fxch	st(1)
				fmul	st(2),st(0)
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 9)]
				fxch	st(3)
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 8)]
				faddp	st(2),st(0)
				fxch	st(1)
				faddp	st(2),st(0)
				faddp	st(1),st(0)
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 4)]
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 5)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 3)]
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 4)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 2)]
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 3)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 1)]
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 2)]
				fld		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem]
				fstp	qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem + (8 * 1)]
				fst		qword ptr [ebp + SEQUENCE.Formant_Tmp_Mem]
				fstp	dword ptr [ebp + SEQUENCE.Sound_Datas]
No_Process_Formant:

No_Process_Filters:
				pop		ecx
				cmp		dword ptr [Play_Type],Play_Song
				jne		Done_With_Song_Instruments
				add		ebp,[Size_Sequence_Dat]
				dec		ecx
				jnz		Set_Song_Instruments
Done_With_Song_Instruments:
				// ----------------------------------
				// Split stereo signal now
				xor		edx,edx
				mov		ebp,[Ptr_Sequence]
				mov		ecx,8
Process_Effects:
				push	edx
				push	ecx
				fld1
				cmp		dword ptr [Play_Type],Play_Instrument
				je		Set_Channel_L_Panning
				fsub	dword ptr [ebp + SEQUENCE.Seq_Pan]
				jmp		Set_Instrument_L_Panning
Set_Channel_L_Panning:
				fsub	dword ptr [Instrument_Pan]
Set_Instrument_L_Panning:
				fsqrt
				fmul	dword ptr [ebp+SEQUENCE.Sound_Datas]
				// Store left signal
				test	edx,edx
				jz		Set_Left_Channel
				fadd	dword ptr [edi]
Set_Left_Channel:
				fstp	dword ptr [edi]
				cmp		dword ptr [Play_Type],Play_Instrument
				je		Set_Channel_R_Panning
				fld		dword ptr [ebp + SEQUENCE.Seq_Pan]
				jmp		Set_Instrument_R_Panning
Set_Channel_R_Panning:
				fld		dword ptr [Instrument_Pan]
Set_Instrument_R_Panning:
				fsqrt
				fmul	dword ptr [ebp + SEQUENCE.Sound_Datas]
				// Store right signal
				test	edx,edx
				jz		Set_Right_Channel
				fadd	dword ptr [edi + 4]
Set_Right_Channel:
				fstp	dword ptr [edi + 4]
				// Take delay arguments
				cmp		dword ptr [Play_Type],Play_Instrument
				je		No_Delay
				mov		edx,[ebp+SEQUENCE.Seq_Cur_Sequence]
				test	edx,edx
				jz		No_Delay_Dsp
				mov		esi,[Cur_Seq_Play_Pos]
				mov		bl,byte ptr [edx + esi + SEQUENCE_DAT.Dsp]
				cmp		bl,SEQ_DSP_DELAY
				jne		No_Delay_Dsp
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas3]
				push	eax
				fild	dword ptr [esp]
				fmul	dword ptr [Flt001]
				fstp	dword ptr [esp]
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas2]
				push	eax
				fild	dword ptr [esp]
				fmul	dword ptr [Flt001]
				fstp	dword ptr [esp]
				movzx	eax,byte ptr [edx + esi + SEQUENCE_DAT.DspDatas1]
				push	eax
				fild	dword ptr [esp]
				fmul	dword ptr [Flt001]
				fstp	dword ptr [esp]
				push	ebp
				call	Set_Delay
No_Delay_Dsp:	// Process delay
				mov		eax,[ebp + SEQUENCE.Seq_Delay_Value]
				test	eax,eax
				jz		No_Delay
				mov		eax,[ebp + SEQUENCE.Seq_Delay_Cursor]
				mov		ebx,[ebp + SEQUENCE.Seq_Delay_Left_Buffer]
				mov		ecx,edi
				fld		dword ptr [ebx + eax * 4]
				fadd	dword ptr [ecx]
				fstp	dword ptr [ecx]
				add		ecx,4
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				fadd	dword ptr [ebx + eax * 4]
				fmul	dword ptr [ebp + SEQUENCE.Seq_Delay_Left_FeedBack_Value]
				fstp	dword ptr [ebx + eax * 4]
				mov		ebx,[ebp + SEQUENCE.Seq_Delay_Right_Buffer]
				fld		dword ptr [ebx + eax * 4]
				fadd	dword ptr [ecx]
				fstp	dword ptr [ecx]
				fld		dword ptr [ebp + SEQUENCE.Sound_Datas]
				fadd	dword ptr [ebx + eax * 4]
				fmul	dword ptr [ebp + SEQUENCE.Seq_Delay_Right_FeedBack_Value]
				fstp	dword ptr [ebx + eax * 4]
				inc		dword ptr [ebp + SEQUENCE.Seq_Delay_Cursor]
				mov		eax,[ebp + SEQUENCE.Seq_Delay_Value]
				cmp		[ebp + SEQUENCE.Seq_Delay_Cursor],eax
				jl		No_Delay
				mov		dword ptr [ebp + SEQUENCE.Seq_Delay_Cursor],0
No_Delay:	

				pop		ecx
				pop		edx
				mov		edx,1
				cmp		dword ptr [Play_Type],Play_Song
				jne		Done_With_Song_Effects
				add		ebp,[Size_Sequence_Dat]
				dec		ecx
				jnz		Process_Effects
Done_With_Song_Effects:
				add		edi,8
				pop		ebp
				pop		edx					// Push is right at the entry point
				dec		edx
				jmp		Mix_Loop
End_Mix_Loop:	pop		edi
				// ----------------------------------
				// Final mixing and conversion into 16 bits output
				mov		edx,[BufferSize]
				shr		edx,1
				lea		esi,Mixing_buffer_Stereo
				mov		ebx,[Ptr_Spectrum_Buffer]
Final_Mix_Loop:	test	edx,edx
				jz		End_Final_Mix_Loop
				lodsd
				mov		[Final_Signal],eax
				// Clip sample
				fld		dword ptr [Final_Signal]
				fld1
				fcom	st(1)
				fnstsw	ax
				sahf
				jae		Min_Value
				fst		dword ptr [Final_Signal]
Min_Value:		fstp	st(0)
				fstp	st(0)
				fld		dword ptr [Final_Signal]
				fld		dword ptr [Fltm1]
				fcom	st(1)
				fnstsw	ax
				sahf
				jbe		Max_Value
				fst		dword ptr [Final_Signal]
Max_Value:		fstp	st(0)
				fstp	st(0)
				fld		dword ptr [Final_Signal]
				fmul	dword ptr [Flt32767]		// Sign the sample
				fistp	dword ptr [Final_Signal]
				fstp	st(0)
				mov		eax,[Final_Signal]
				test	eax,eax
				jns		Sign_Sample
				dec		ax
Sign_Sample:	//cmp		dword ptr [Render_Wave],0
				//jne		No_Store_Spectrum
				mov		[ebx],ax					// Left
				add		ebx,2
//No_Store_Spectrum:
				stosw
				dec		edx
				jmp		Final_Mix_Loop
End_Final_Mix_Loop:	
				mov		[Ptr_Spectrum_Buffer],ebx
			}
			break;
	}
}

// ------------------------------------------------------
// Name: Get_Positions_Number()
// Desc: Retrieve the number of positions
int __stdcall Get_Positions_Number(void) {
	return(Positions[0]->Get_Number_Of_Entries());
}

// ------------------------------------------------------
// Name: Get_Position_Dat()
// Desc: Retrieve the address of a position
int __stdcall Get_Position_Dat(int Pos_Number, int Pos_Index) {
	if(Pos_Index > Positions[Pos_Number]->Get_Number_Of_Entries() - 1) return(NULL);
	return(Positions[Pos_Number]->Get_Entry_By_Ordinal_Long(Pos_Index));
}

// ------------------------------------------------------
// Name: Get_Sequence_Dat()
// Desc: Retrieve the address of a sequence
int __stdcall Get_Sequence_Dat(int Seq_Number) {
	if(Seq_Number > Sequences.Get_Number_Of_Entries() - 1) return(NULL);
	return(Sequences.Get_Entry_By_Ordinal_Long(Seq_Number));
}

// ------------------------------------------------------
// Name: Get_Instrument_Dat()
// Desc: Retrieve the address of an instrument
int __stdcall Get_Instrument_Dat(LPINSTRUMENT Cur_Instrument, int Instr_Number) {
	if(!Cur_Instrument) return(0);
	if(Instr_Number > Cur_Instrument->Datas->Get_Number_Of_Entries() - 1) return(0);
	return(Cur_Instrument->Datas->Get_Entry_By_Ordinal_Long(Instr_Number));
}

// ------------------------------------------------------
// Name: Get_Instrument_Comb()
// Desc: Retrieve the number of combinators of an intrument
int __stdcall Get_Instrument_Comb(LPINSTRUMENT Cur_Instrument) {
	if(!Cur_Instrument) return(0);
	return(Cur_Instrument->Datas->Get_Number_Of_Entries());
}

// ------------------------------------------------------
// Name: Get_Instrument()
// Desc: Retrieve the address of an instrument
int __stdcall Get_Instrument(int Instr_Number) {
	if(Instr_Number > Instruments.Get_Number_Of_Entries() - 1) return(0);
	return(Instruments.Get_Entry_By_Ordinal_Long(Instr_Number));
}

// ------------------------------------------------------
// Name: Wave_WhiteNoise()
// Desc: Produce a whitenoise
void __stdcall Wave_WhiteNoise(void) {
	_asm {
			mov		eax,[Random_Seed]
			inc		eax
			imul	eax,31415621
			mov		[Random_Seed],eax
			and		eax,0xffff
			sub		eax,0x8000
			push	eax
			fild	dword ptr [esp]
			pop		eax
			fdiv	dword ptr [Flt32768]
	}
}

// ------------------------------------------------------
// Name: Get_Mix_Buffer_Len()
// Desc: Convert a given BPM into ticks
int __stdcall Get_Mix_Buffer_Len(DWORD BPM) {
	int Ticks;

	_asm {
			push	ebx
			push	edx
			mov		eax,44100
			mov		ebx,10
			imul	eax,ebx
			xor		edx,edx
			mov		ebx,[BPM]
			test	ebx,ebx
			jz		DSZeroBPM
			idiv	ebx
DSZeroBPM:	shr		eax,2
			pop		edx
			pop		ebx
			mov		[Ticks],eax
	}
	return(Ticks);
}

// ------------------------------------------------------
// Name: Set_Delay()
// Desc: Set the delay effect values
void __stdcall Set_Delay(LPSEQUENCE Seq_Dats, float Delay_Distance, float Delay_Feedback, float Delay_Pan) {
	_asm {
			mov		eax,[Seq_Dats]
			fld		dword ptr [Delay_Distance]
			fimul	dword ptr [Max_Buffer]
			fistp	dword ptr [eax+SEQUENCE.Seq_Delay_Value]
			fld1
			fsub	dword ptr [Delay_Pan]
			fsqrt
			fmul	dword ptr [Delay_Feedback]
			fstp	dword ptr [eax+SEQUENCE.Seq_Delay_Left_FeedBack_Value];
			fld		dword ptr [Delay_Pan]
			fsqrt
			fmul	dword ptr [Delay_Feedback]
			fstp	dword ptr [eax+SEQUENCE.Seq_Delay_Right_FeedBack_Value];
	}
}

// ------------------------------------------------------
// Name: Set_Reverb()
// Desc: Set the reverb effect values
void __stdcall Set_Reverb(LPSEQUENCE Seq_Dats, float Reverb_Distance, float Reverb_Feedback, float Reverb_Pan) {
	_asm {
			mov		eax,[Seq_Dats]
			fld		dword ptr [Reverb_Distance]
			fimul	dword ptr [Max_Buffer]
			fistp	dword ptr [eax+SEQUENCE.Seq_Reverb_Value]
			fld1
			fsub	dword ptr [Reverb_Pan]
			fsqrt
			fmul	dword ptr [Reverb_Feedback]
			fstp	dword ptr [eax+SEQUENCE.Seq_Reverb_Left_FeedBack_Value];
			fld		dword ptr [Reverb_Pan]
			fsqrt
			fmul	dword ptr [Reverb_Feedback]
			fstp	dword ptr [eax+SEQUENCE.Seq_Reverb_Right_FeedBack_Value];
	}
}

// ------------------------------------------------------
// Name: Clamp_Value()
// Desc: Clamp the a value between [0..1] 
void __stdcall Clamp_Value(void) {
	_asm {
			fcom	dword ptr [Flt1]
			fnstsw	ax
			sahf
			jb		Clamp_Volume_Max
			fstp	st(0)
			fld1
Clamp_Volume_Max:
			ftst
			fnstsw	ax
			sahf
			ja		Clamp_Volume_Min
			fsub	st(0),st(0)
Clamp_Volume_Min:
	}
}

// ------------------------------------------------------
// Name: Clear_Buffers()
// Desc: Clear the mixing buffers
void Clear_Buffers(void) {
	memset(&Mixing_buffer_Stereo, 0, (44100 * 2) * 4);
	memset(&Spectrum_Buffer, 0, 44100 * 2);
	memset(Dummy_Sequence_Dats.Seq_Delay_Left_Buffer, 0, 44100 * 4);
	memset(Dummy_Sequence_Dats.Seq_Delay_Right_Buffer, 0, 44100 * 4);
	memset(Dummy_Sequence_Dats.Seq_Reverb_Left_Buffer, 0, 44100 * 4);
	memset(Dummy_Sequence_Dats.Seq_Reverb_Right_Buffer, 0, 44100 * 4);

	for(int i = 0; i < 8; i++) {
		memset(Sequences_Dats[i].Seq_Delay_Left_Buffer, 0, 44100 * 4);
		memset(Sequences_Dats[i].Seq_Delay_Right_Buffer, 0, 44100 * 4);
		memset(Sequences_Dats[i].Seq_Reverb_Left_Buffer, 0, 44100 * 4);
		memset(Sequences_Dats[i].Seq_Reverb_Right_Buffer, 0, 44100 * 4);
	}
}

// ------------------------------------------------------
// Name: Clear_Sequence_Dats()
// Desc: Clear the datas of a sequence
void Clear_Sequence_Dats(LPSEQUENCE Seq_To_Clear) {
	Seq_To_Clear->Seq_Position_Volume = 1.0f;
	Seq_To_Clear->Seq_Pitch_Bend_Freq = 0;
	Seq_To_Clear->Seq_Current_Freq = 0;
	Seq_To_Clear->Seq_Mute_Volume = 1.0f;
	Seq_To_Clear->Seq_Position_Transpose = 0;
	Seq_To_Clear->Seq_Vol_Side = 0;
	Seq_To_Clear->Seq_Old_Vol_Side_Up = 0;
	Seq_To_Clear->Seq_Old_Vol_Side_Down = 0;
	Seq_To_Clear->Seq_Arpeggio = 0;
	Seq_To_Clear->Seq_Old_Arpeggio = 0;
	Seq_To_Clear->Seq_Vibrato = 0;
	Seq_To_Clear->Seq_Vibrato_Freq = 0;
	Seq_To_Clear->Seq_Vibrato_Tick = 0;
	Seq_To_Clear->Seq_Old_Vibrato = 0;
	Seq_To_Clear->Seq_Pitch_Bend = 0;
	Seq_To_Clear->Seq_Tone_Portamento_Speed = 0;
	Seq_To_Clear->Seq_Tone_Portamento_Note = 0;
	Seq_To_Clear->Seq_Old_Tone_Portamento_Speed = 0;
	Seq_To_Clear->Seq_Old_Tone_Portamento_Note = 0;
	Seq_To_Clear->Disto_Param = 0;
	Seq_To_Clear->Formant_Param1 = 0;
	Seq_To_Clear->Seq_Hi_Pass_Param1 = 0;
	Seq_To_Clear->Seq_Hi_Pass_Param2 = 0;
	Seq_To_Clear->Seq_Hi_Pass_In1 = 0;
	Seq_To_Clear->Seq_Hi_Pass_Out1 = 0;
	Seq_To_Clear->Seq_Hi_Pass_In2 = 0;
	Seq_To_Clear->Seq_Hi_Pass_Out2 = 0;
	Seq_To_Clear->Seq_Lo_Pass_Param1 = 0;
	Seq_To_Clear->Seq_Lo_Pass_Param2 = 0;
	Seq_To_Clear->Seq_Lo_Pass_In1 = 0;
	Seq_To_Clear->Seq_Lo_Pass_Out1 = 0;
	Seq_To_Clear->Seq_Lo_Pass_In2 = 0;
	Seq_To_Clear->Seq_Lo_Pass_Out2 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Param1 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Param2 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Out1 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Out2 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Out3 = 0;
	Seq_To_Clear->Seq_VCF_Pass_Out4 = 0;
	Seq_To_Clear->Seq_VCF_Pass_In1 = 0;
	Seq_To_Clear->Seq_VCF_Pass_In2 = 0;
	Seq_To_Clear->Seq_VCF_Pass_In3 = 0;
	Seq_To_Clear->Seq_VCF_Pass_In4 = 0;
	Seq_To_Clear->Seq_Current_Instrument_Dat = 0;
	Set_Delay(Seq_To_Clear, 0.0f, 0.0f, 0.0f);
	Seq_To_Clear->Seq_Delay_Cursor = 0;
	Set_Reverb(Seq_To_Clear, 0.0f, 0.0f, 0.0f);
	Seq_To_Clear->Seq_Reverb_Cursor = 0;
	for(int i = 0; i < 256; i++) {
		memset(&Seq_To_Clear->Seq_Cur_Instrument_Vars[i], 0, sizeof(INSTRUMENT_VARS));
	}
	Seq_To_Clear->Formant_Tmp_Mem[0] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[1] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[2] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[3] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[4] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[5] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[6] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[7] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[8] = 0;
	Seq_To_Clear->Formant_Tmp_Mem[9] = 0;
}
