#define WIN32_LEAN_AND_MEAN             // Exclude rarely-used stuff from Windows headers

#include <windows.h>
#include <memory.h>
#include <intrin.h>
#include <stdlib.h>
#include <math.h>

#include "FM8_Utils.h"
#include "FM8_Synth.h"

int seed = 0;

float param[3];
float Average1 = 0;
float Average2 = 0;

extern "C" void __cdecl _CIfmod()
{
    _asm    fxch    st(1)           // Swap arguments
    _asm  __CIfmod1:      
    _asm    fprem                   // Get the partial remainder
    _asm    fstsw   ax              // Get coprocessor status
    _asm    test    ax,0400h        // Complete remainder ?
    _asm    jnz     __CIfmod1       // No, go get next remainder
    _asm    fstp    st(1)           // Set new top of stack
}

#ifdef _DEBUG
	void CheckForNAN(float x) 
	{
		if (!((x == x) && (x <= FLT_MAX && x >= -FLT_MAX)))
			DebugBreak();
	}
#else
	#define CheckForNAN()
#endif

void FM8_VariableStateFilter::Setup(float CutOffFreq, float SampleRate, float Resonance)
{
    CutOffFreq = min(CutOffFreq, SampleRate / 4);
    Resonance  = FM8_powf(Resonance / 100.f, 1.5f) * (param[1] + 0.5f);

    Freq = 2.0f*FM8_sinf(M_PI*MIN(0.24f, CutOffFreq/(SampleRate*2))); // the fs*2 is because it's double sampled*/
    Damp = MIN(2.0f*(1.0f - FM8_powf(Resonance, 0.25f)), MIN(1.9f, 2.0f/Freq - Freq*0.5f));
}

void FM8_VariableStateFilter::ExecuteFilter(float input)
{
    for (int i = 0; i < 2; i++)
    {
        float AmplifiedInput = (input + PrevSample) ;
        PrevSample = input;
        
        if (input >  1.1f) input =  1.1f;
        if (input < -1.1f) input = -1.1f;
        if (Low   >  1.1f) Low   =  1.1f;
        if (Low   < -1.1f) Low   = -1.1f;
        if (Band  >  1.1f) Band  =  1.1f;
        if (Band  < -1.1f) Band  = -1.1f;

        Notch = AmplifiedInput - Damp*Band;
        Low = Low + Freq*Band;
        High = Notch - Low;
        Band = Freq*High + Band - (param[2]*0.01f)*Band*Band*Band;
    }
}

#define WAVEFORMS_COUNT 8
bool  bLUTInit = false;
float Waveforms[WAVEFORMS_COUNT][0x10000];
float TempBuffer[0x10000];

// Smoothing kernel computation
#define COEF_SCALING 600
float coef[11] = {0.0145985401459854f, 0.0306299603261906f, 0.0726007632458986f, 0.124479528724904f, 0.166450331644612f, 0.182481751824818f, 0.166450331644612f, 0.124479528724904f, 0.0726007632458986f, 0.0306299603261906f, 0.0145985401459854f};
float Kernel[11 * COEF_SCALING];

float SmoothStep(float t, float smooth)
{
    return (FM8_expf(abs(t) * smooth) - 1) / (FM8_expf(smooth) - 1) * SIGN(t);
}

void FM8_GenerateWaveTable()
{
    // Make sure this method runs only once
    if (bLUTInit) return ;
    bLUTInit = true;

    for (int i=0;i<0x10000;i++)
    {
        float factor = i / (float)0xFFFF;
        float factor_05 = factor - 0.5f;
		float sin = FM8_sinf(2 * M_PI * factor);

        // 0: Sin wave
        Waveforms[0][i] = sin;
        
        // 1: Parabol
        Waveforms[1][i/2] = -4*factor_05*factor_05 + 1;
        Waveforms[1][0x8000 + i/2] = -Waveforms[1][i/2];

        // 2: Triangle
        Waveforms[2][i] = abs(factor_05) * 4 - 1;

        // 3: Square
        Waveforms[3][i] = i > 0x8000 ? +0.8f : -0.8f;

		// 4: Sawtooth
        Waveforms[4][i] = 1 - 2 * factor;

		// 5: Soft-Square
        Waveforms[5][i] = SmoothStep(sin, -5);

		// 6: Soft tristate 
		// 7: Short tristate 
		Waveforms[7][i] = 0;
        Waveforms[7][i / 2] = Waveforms[6][i] = SmoothStep(sin*sin*sin*sin*sin*sin*sin, -5);

        // 1 + 3 + 5 Square 
        //float OscOutput = Math.Sin(2 * Math.PI * OscInput) + Math.Sin(2 * Math.PI * OscInput * 3) / 3 + Math.Sin(2 * Math.PI * OscInput * 5) / 5;

         // 1 + 2 + 3 Saw 
        // float OscOutput = Math.Sin(2 * Math.PI * OscInput) +Math.Sin(2 * Math.PI * OscInput * 2) / 2 + Math.Sin(2 * Math.PI * OscInput * 3) / 3;
    }

#ifndef SKIP_WAVEFORM_SMOOTHING

	// Build waveform smoothing kernel
    for (int j=0;j<10;j++)
        for (int i=0;i<COEF_SCALING;i++)
            Kernel[j*COEF_SCALING + i] = (coef[j] * (1 - i / (float)COEF_SCALING) + coef[j+1] * (i / (float)COEF_SCALING)) / COEF_SCALING;

    // Smooth wave forms
    for (int k=3;k<=5;k++)
    {
        for (int j=0;j<0x10000;j++)
        {
            TempBuffer[j] = 0.f;
            for (int i=0;i<(10 * COEF_SCALING);i++)
                TempBuffer[j] += Waveforms[k][(unsigned short)(j + i - 5 * COEF_SCALING)] * Kernel[i];
        }
        FM8_memcpy(Waveforms[k], TempBuffer, 0x10000 * 4);
    }
#endif
}

float FM8_Synth::GetScaleValue(int OpIndex, float NoteIndex)
{
    // OP 1 KeySc:
    // Note:  0.00000 58.00000  96.00000 127.00000 
    // Level: 0.00000  0.00000 -14.76923 -39.38462 
    // Slope: 0.50000  0.50000   0.41000   0.81000 
    
    // Search for segment
    int Seg_Index = 0;
    while (NoteIndex > (int)OP_KeySc_Note[OpIndex][Seg_Index + 1])
        Seg_Index++;

    // Compute Interpolator
    float SegmentSlope = (0.5001f - OP_KeySc_Slope[OpIndex][Seg_Index + 1]) * 20;

    // Compute point in segment 
    float CurrKeyScale    = OP_KeySc_Note[OpIndex][Seg_Index];
    float NextKeyScale    = OP_KeySc_Note[OpIndex][Seg_Index + 1];
    float OffsetInSegment = (NoteIndex - CurrKeyScale) / (NextKeyScale - CurrKeyScale);
    float Interpolator    = (FM8_expf(OffsetInSegment * SegmentSlope) - 1) / (FM8_expf(SegmentSlope) - 1);

    return OP_KeySc_Level[OpIndex][Seg_Index] +
           (OP_KeySc_Level[OpIndex][Seg_Index + 1] -
            OP_KeySc_Level[OpIndex][Seg_Index]) * Interpolator;
}

float FM8_Synth::GetEnvValue(int OpIndex, float Time_PressToNow, float Time_PressToRelease, bool& bActive)
{
    // Sample envelope data:
    //   Level:  1.00000 0.50000 0.10000 
    //   R-Time: 0.10000 0.70000 0.20000 
    //   A-Time: 0.00000 0.10000 0.80000 
    //   Slope:  0.50000 0.90000 0.90000 
    //
    //   OP_Env_SustainLoopStart:      0  
    //   OP_Env_SustainLoopEnd:        1  
    // Get sustain loop time
    int ValuesCount = OP_Env_Length[OpIndex] - 1;
    int LoopStartIndex = OP_Env_SustainLoopStart[OpIndex] + 1;
    int LoopEndIndex = MIN(OP_Env_SustainLoopEnd[OpIndex] + 1, ValuesCount);
    float LoopStartTime = pcOP_Env_AbsTime[OpIndex][LoopStartIndex];
    float LoopEndTime = pcOP_Env_AbsTime[OpIndex][LoopEndIndex];
    bool bLoopExists = (LoopStartIndex != LoopEndIndex);

    // Local members
    float t;
    float PrevLevel = 0;
    int Seg_Index;

    // If key was release 
    if (Time_PressToRelease > 0)
    {
        // Store release value 
        if (csEnvRelease[OpIndex] == FLT_MAX)
            csEnvRelease[OpIndex] = GetEnvValue(OpIndex, Time_PressToRelease, 0, bActive);

        Seg_Index = LoopEndIndex;
        PrevLevel = csEnvRelease[OpIndex]; 
        t = LoopEndTime + (Time_PressToNow - Time_PressToRelease);
    }
    else
    {
        t = Time_PressToNow;

        // Inside loop?
        if ((t >= LoopStartTime) && (bLoopExists))
        {
            // Set segment search to loop start
            Seg_Index = LoopStartIndex;

            // Overlapped ?   
            if (t >= LoopEndTime)
            {
                // If overlapped, asuume prev level is the end of the loop
                PrevLevel = OP_Env_Level[OpIndex][LoopEndIndex - 1];
            }
            else
            {
                PrevLevel = OP_Env_Level[OpIndex][LoopStartIndex - 1];
            }

            
            t = LoopStartTime + fmod(t - LoopStartTime, LoopEndTime - LoopStartTime);
        }
        else
        {
            // Set segment search to start
            Seg_Index = 0;

            // If before the loop, assume prev level is end level (don't know why this works like this)
            PrevLevel = OP_Env_Level[OpIndex][ValuesCount];
        }
    }

    // Find Segment
    // At this point "t" should be smaller than the absolute time of the segment Seg_Index.
    // We're search for the last segment that DO NOT over shoots "t"
    while ((Seg_Index < ValuesCount) && (t > pcOP_Env_AbsTime[OpIndex][Seg_Index + 1]))
    {
        PrevLevel = OP_Env_Level[OpIndex][Seg_Index];
        Seg_Index++;
    }

    // Exit if after envelope
    //if (Seg_Index >= ValuesCount)
    if (t >= pcOP_Env_AbsTime[OpIndex][ValuesCount + 1])
    {
        bActive = false;
        return OP_Env_Level[OpIndex][ValuesCount];
    }

    // Compute Interpolator
    float SegmentSlope = (0.5001f - OP_Env_Slope[OpIndex][Seg_Index]) * 15;

    // Compute time in segment
    float SegmentWidth = OP_Env_RelTime[OpIndex][Seg_Index];
    float TimeInSegment = t - pcOP_Env_AbsTime[OpIndex][Seg_Index];
    if (TimeInSegment > SegmentWidth) TimeInSegment = SegmentWidth;
    if (SegmentWidth != 0) TimeInSegment /= SegmentWidth;

    // Compute interpolator and final vlaue
    float Interpolator = (FM8_expf(TimeInSegment * SegmentSlope) - 1) / (FM8_expf(SegmentSlope) - 1);
    //Interpolator = LUT1[(int)(65535 * TimeInSegment * SegmentSlope)] * LUT1[(int)(65535 * SegmentSlope)];
    float EnvValue     = PrevLevel + (OP_Env_Level[OpIndex][Seg_Index] - PrevLevel) * Interpolator;

    return EnvValue;
}

void FM8_Synth::SetupInstrument()
{
    // compute Envelope AbsTime
    for (int i = 0; i < 9; i++)
        for (int j = 0; j < OP_Env_Length[i]; j++)
            pcOP_Env_AbsTime[i][j + 1] = pcOP_Env_AbsTime[i][j] + OP_Env_RelTime[i][j];

    Note_Velocity = 1;

    // Scale FM-Matrix
    for (int i = 0; i < 8*8; i++)
        pcFM_Matrix[0][i] = FM_Matrix[0][i] * FM_Matrix[0][i] * (2.113f / 10000);

    for (int i = 0; i < 8; i++)
    {
        // Scale self feedback
        pcFM_Matrix[i][i] *= 0.5f;

        // scale OP-X inputs
        pcFM_Matrix[6][i] *= (1 / 2.113f);

        // scale offset by sample rate
        OP_Offset[i] /= SAMPLE_RATE;
    }

    // Rescale OP-X feedbacks
    for (int i = 0; i < 6; i++)
        pcFM_Matrix[i][6] *= 0.75f;

    // Compute the OpX cut-off frequancy
    float OpX_NoiseCutOff = FM8_powf(10.f, OPX_NoiseCutOff / 24.51472341f);
    OpX_VSF.Setup(OpX_NoiseCutOff, SAMPLE_RATE, OPX_NoiseReso);
    pcOpX_NoiseAmp = (782 / FM8_expf(0.0822f * OPX_NoiseCutOff)) * (OPX_NoiseAmp * OPX_NoiseAmp / 10000.f);

    // Precompute Transp
    pcPitch_Transp = FM8_powf(2.f, Pitch_Transp / 12.f);

    // Set Default velocity
    SetVelocity(0x80 / 127.f);

    // Set all operators to silence (no note was played - yet)
    ActiveOperators = 0;	
}


void FM8_Synth::RenderToBuffer(float* buffer, int bufferLengthInSamples)
{
    
    //Exit if voice is ideal
    if ((Note_Index == -1) || (ActiveOperators == 0))
        return;

    // Precompute things
    float PitchEnvScale = (2.8291f / 10000) * Pitch_Envelope * Pitch_Envelope;

    // Local members
    float PitchScale;

    // Setup Base Freqs
    float BaseOscFreq[6];
    float BaseFreq = FM8_powf(2.f, (Note_Index - 69) / 12.f) * 440;
    for (int row = 0; row < 6; row++)
        BaseOscFreq[row] = BaseFreq * OP_Ratio[row] * pcPitch_Transp / SAMPLE_RATE;
    
    // For each stereo-sample in buffer 
    for (int index = 0; index < bufferLengthInSamples; index++)
    {
        SampleCounter++;

        // Update Envelopes
        for (int i = 0; i < 10; i++)
            EnvelopeCurrent[i] += EnvelopeIncrement[i];

        // Compute Envelopes slope
        if ((index & 63) == 0)
        {
            // Increment envelope times
            TimeFromPress = (float)SampleCounter / SAMPLE_RATE;

            // Reset Active operators (channel 8 allways exists, and will make this a zero)
            ActiveOperators = -1;

            // compute envelopsw
            for (int i = 0; i < 9; i++)
            {
                // Skip non-active evelopes (apart from pitch envelope)
                if (!EnvelopeActive[i] && (i != 8)) 
                {
                    EnvelopeIncrement[i] = 0;
                    continue;
                }

                // Compute envelope target value
                float EnvelopeTarget = GetEnvValue(i, TimeFromPress * csTimeScale[i], TimeRelease * csTimeScale[i], EnvelopeActive[i]);
                EnvelopeIncrement[i] = (EnvelopeTarget - EnvelopeCurrent[i]) / 64;
                ActiveOperators++;

                // Compute pitch envelope target
                if (i == 8)
                {
                    float PitchEnvTarget = FM8_expf(PitchEnvScale * EnvelopeTarget);
                    EnvelopeIncrement[9] = (PitchEnvTarget - EnvelopeCurrent[9]) / 64;
                }				
            }

            // Check if terminated
            if (ActiveOperators == 0)
                return;
        }

        // Compute Pitch scaling
        PitchScale = EnvelopeCurrent[9];

        // Process operators
        float TotalOutputL = 0;
        float TotalOutputR = 0;
        for (int row = 0; row < 8; row++)
        {
            // Check if operator is non-active
            if (!EnvelopeActive[row])
                continue;

            // Collect inputs and feedbacks
            float OpInput = 0;
            for (int i = 0; i < 8; i++)
                if (FMMatrixValue[i] != 0)
                    OpInput += pcFM_Matrix[row][i] * FMMatrixValue[i];
            
            float OscOutput = 0;
            if (row < 6)
            {	
                // Compute Osc state
                float OscFreq = OscPos[row] + BaseOscFreq[row] * PitchScale + OP_Offset[row];
                OscPos[row] = OscFreq - (int)OscFreq;

                // Compute Osc new state (adjusted by input)
                OscOutput = Waveforms[OP_Waveform[row]][(unsigned short)(65535 * (OscPos[row] + OpInput))];
            }
            else if (row == 6) // X Op
            {
                // Generate noise and low pass it
                OpX_VSF.ExecuteFilter(FM8_sfrand(&seed));
                OscOutput = OpInput + OpX_VSF.Low * pcOpX_NoiseAmp;
                
                // Gain (Input +/-1  Output +/-3.2515)
                float Gain = 0.0003f * OPX_SatGain * OPX_SatGain + 0.0006f * abs(OPX_SatGain) + 0.1915f;
                OscOutput *= Gain;

                // Wave shaper (http://www.musicdsp.org/showone.php?id=86)
                float Level = 0.0332f * FM8_expf(0.0457f * OPX_SatLimit);

                // Scale and remove sign
                int Sign = SIGN(OscOutput);
                OscOutput = abs(OscOutput) / Level;

                // Perofrm clipping
                if (OscOutput > 1) OscOutput = 1;

                // Soft Clipping
                OscOutput = -OscOutput * OscOutput  + 2 * OscOutput;

                // Rescale level
                OscOutput *= Sign * Level;

                // Clamp DC
                Average1 = Average1 * 0.99999f + OscOutput * 0.00001f;
                OscOutput -= Average1;
            }
            else // Z Op
            {
                // first multiband
                float Stage1Out;
                OpZ_S1_VSF.ExecuteFilter(OpInput*4);
                if (OPZ_Mode1 < 50)
                    Stage1Out = MIX100(OpZ_S1_VSF.Low, OpZ_S1_VSF.Band, OPZ_Mode1 * 2);
                else
                    Stage1Out = MIX100(OpZ_S1_VSF.Band, OpZ_S1_VSF.High, (OPZ_Mode1 - 50) * 2);

                // second multiband
                float Stage2Out;
                OpZ_S2_VSF.ExecuteFilter(MIX100(OpInput, Stage1Out, OPZ_Routing));
                if (OPZ_Mode2 < 50)
                    Stage2Out = MIX100(OpZ_S2_VSF.Low, OpZ_S2_VSF.Band, OPZ_Mode2 * 2);
                else
                    Stage2Out = MIX100(OpZ_S2_VSF.Band, OpZ_S2_VSF.High, (OPZ_Mode2 - 50) * 2);

                // mix
                OscOutput = MIX100(Stage1Out, Stage2Out, OPZ_Mix);				
            }

			// Check that output is a number (debug only)
			CheckForNAN(OscOutput);

			// Compute output value
            FMMatrixValue[row] = OscOutput * EnvelopeCurrent[row] * csKeyScaleAmp[row] * csVelocityAmp[row];

            // Add operator to output
            TotalOutputL += FMMatrixValue[row] * pcOP_OutVol[0 + row];
            TotalOutputR += FMMatrixValue[row] * pcOP_OutVol[9 + row];
        }

        // Store value
        buffer[index * 2 + 0] += TotalOutputR;
        buffer[index * 2 + 1] += TotalOutputL;
    }
}

void FM8_Synth::SetVelocity(float Velocity)
{
    Note_Velocity = Velocity;

    for (int row = 0; row < 9; row++)
    {
        csVelocityAmp[row] = 1;

        // Compute velocity amplification
        float OpVelocity = OP_Velocity[row];
        float VelAmp;
        if (OpVelocity > 0)
            VelAmp = 0.025f * FM8_log2f(Note_Velocity) + 0.011f;
        else
            VelAmp = 0.0274f * Note_Velocity * Note_Velocity + 0.0088f * Note_Velocity - 0.0159f;

        csVelocityAmp[row] = FM8_expf(OpVelocity * VelAmp);
    }
}

void FM8_Synth::PlayNote(int NoteIndex, int VoiceIndex)
{
    // Reset Variables
    ActiveOperators = -1;
    Note_Index = NoteIndex;
    TimeFromPress = 0;
    TimeRelease = 0;
    SampleCounter = 0;
    FM8_memset(EnvelopeCurrent, 0, 10*4);
    FM8_memset(EnvelopeIncrement, 0, 10*4);

    // Precompute Transp
    float Detune = 6E-10f * Unison_Detune * Unison_Detune * Unison_Detune * Unison_Detune;
    pcPitch_Transp = FM8_powf(2.f, Pitch_Transp / 12.f) * (1 + Detune * VoiceIndex);

    // Unison Pan
    int UnisonPanL = 100 + Unison_Pan;
    int UnisonPanR = 100 - Unison_Pan;
    if (VoiceIndex & 1)
    {
        UnisonPanL = 100 - Unison_Pan;
        UnisonPanR = 100 + Unison_Pan;
    }

    // Randomize each ocilator start position
    for (int row = 0; row < 6; row++)
        OscPos[row] = OP_KeySync[row] ? 0 : FM8_frand(&seed);

    for (int row = 0; row < 9; row++)
    {
        csEnvRelease[row]  = FLT_MAX;
        csKeyScaleAmp[row] = 1;
        csTimeScale[row]   = 1;

        // Precompute output gaining 
        float OutAmp = OP_OutVol[row] * OP_OutVol[row] * (0.111f / 10000);
        pcOP_OutVol[0+row] = OutAmp * (100 + OP_OutPan[row]) * UnisonPanL / 10000.f * Gain;
        pcOP_OutVol[9+row] = OutAmp * (100 - OP_OutPan[row]) * UnisonPanR / 10000.f * Gain;

        // Skip if not used
        EnvelopeActive[row] = (OP_KeyScale_Length[row] != 0);
        if (!EnvelopeActive[row])
            continue;

        // Compute Keyscales
        csKeyScaleAmp[row] = FM8_expf(9.210340372f * ((GetScaleValue(row, Note_Index) + 80) / 80)) / 10000;

        // Compute Time scaling (optional)
        csTimeScale[row] = 1 / FM8_expf((48 - Note_Index) * 0.000048f * pcPitch_Transp * OP_KeyScale[row]);
    }

    // Compute the OpZ cut-offs frequancy
    if (EnvelopeActive[7])
    {
        float OpZ_BaseScale = 30.66f * FM8_expf(0.06f * GetScaleValue(7, Note_Index));
        float OpZ_CutOff1 = FM8_expf(0.06f * OPZ_CutOff)                    * OpZ_BaseScale;
        float OpZ_CutOff2 = FM8_expf(0.06f * (OPZ_CutOff + OPZ_Spread / 2)) * OpZ_BaseScale;
        OpZ_S1_VSF.Setup(OpZ_CutOff1, SAMPLE_RATE, OPZ_Reso1);
        OpZ_S2_VSF.Setup(OpZ_CutOff2, SAMPLE_RATE, OPZ_Reso2);
        csKeyScaleAmp[7] = 0.18f;
    }
}

void FM8_Synth::ReleaseNote()
{
    if (TimeRelease == 0)
        TimeRelease = TimeFromPress + 0.000001f;
}


int FM8_LoadInstruments(UInt8* pBlock, FM8_Synth*& pInstruments)
{
    // Build LUTs if needed
    FM8_GenerateWaveTable();

    // Get the number of instruments
    UInt8 nInstruments = LoadInterlacedStruct(pBlock, (UInt8*&)pInstruments, sizeof(FM8_Synth), 250);
        
    // load dynamic blocks
    for (int i = 0; i < 6; i++)
    {
        Int8 ItemSize     = i < 2 ? 1 : 4; 

        for (int j = 0; j < nInstruments; j++) 
        {
            // Load array
            FM8_Synth* DstInst = pInstruments+j;
            Int8* JaggedLength = i < 3 ? DstInst->OP_KeyScale_Length : DstInst->OP_Env_Length;
            UInt8* pDst = ((UInt8*)DstInst) + 250 + (9 * 128 * 4) * i;
            Load2DJaggedArray(pDst, pBlock, ItemSize, JaggedLength);
        }
    }

    for (int j = 0; j < nInstruments; j++) 
        (pInstruments + j)->SetupInstrument();

    return nInstruments;
}

float* FM8_Waveform(int WaveformIndex)
{
	return Waveforms[WaveformIndex];
}

void FM8_Synth::RenderEnvelope(int OpIndex, float FromT, float ToT, int Samples, float* pBuffer)
{
    int ValuesCount = OP_Env_Length[OpIndex] - 1;
    int LoopEndIndex = MIN(OP_Env_SustainLoopEnd[OpIndex] + 1, ValuesCount);
    float LoopEndTime = pcOP_Env_AbsTime[OpIndex][LoopEndIndex];

	// Store and reset release value
	float StoreEnvRelease = csEnvRelease[OpIndex];
	csEnvRelease[OpIndex] = FLT_MAX;

	bool bActive;
	float Step = (ToT - FromT) / Samples;
	for (int i=0; i<Samples; i++)
	{
		float Time = FromT + Step * i;
		float Release = Time > LoopEndTime ? LoopEndTime : 0;
		*(pBuffer++) = GetEnvValue(OpIndex, Time, Release, bActive);
	}

	// Restore release
	csEnvRelease[OpIndex] = StoreEnvRelease;
}